/****************************************************************************

   Copyright (c) 2000, Western Avionics LTD.             All Rights Reserved.

*****************************************************************************

   File          : sbiib.h

   Author        : S.J.Buckley

   Date          : 11-01-2000

   Function      : Main include file for SBIIB drivers

   Update        : dd-mm-yy
   By            : S.J.Buckley
   Reason        :

****************************************************************************/

#include <stdio.h>
#include "sb_flags.h"
#include "sb_error.h"

/****************************************************************************
                            Drivers Rev Info
****************************************************************************/
#define  SBIIB_DRVS_VERSION_INFO  "1LxxxxxHxx Rev 1.0\0"
#define  SBIIB_DRVS_VERSION_SIZE  19

/****************************************************************************
 Variable type definitions
****************************************************************************/

#if SBIIB_INT_16_BITS
typedef unsigned char   Ubyte;
typedef char            Sbyte;
typedef unsigned int    Uword;
typedef int             Sword;
typedef unsigned long   Ulong;
typedef long            Slong;
typedef double          Sdoub;
#else
typedef unsigned char   Ubyte;
typedef char            Sbyte;
typedef unsigned short  Uword;
typedef short           Sword;
typedef unsigned int    Ulong;
typedef int             Slong;
typedef double          Sdoub;
#endif

typedef Uword           Error;

#define NO_POINTER      NULL

/****************************************************************************
                    Hardware map for registers and queues
****************************************************************************/
/*
 Base registers
*/
#define   CONTROL_REG               (Ulong)0x0000
#define   CLOCK_ADDR                (Ulong)0x0000
#define   COMMAND_REG               (Ulong)0x0004
#define   STATUS_REG                (Ulong)0x0006
#define   BACKGROUND_REG            (Ulong)0x0008
#define   RT_SIM_PTR                (Ulong)0x000A
#define   HI_PRI_Q_PTR              (Ulong)0x000C
#define   LO_PRI_Q_PTR              (Ulong)0x000E
#define   BIN_Q_PTR                 (Ulong)0x0010
#define   BIN_Q_EVENT               (Ulong)0x0012
#define   VME_IRQ_1                 (Ulong)0x0014
#define   VME_IRQ_2                 (Ulong)0x0016
#define   IRQ_SELECTION             (Ulong)0x0018
#define   GAP_TIME_REG              (Ulong)0x001A
#define   RT_RESPONSE_TIME          (Ulong)0x001C
#define   RT_RESPONSE_TIMEOUT       (Ulong)0x001E
#define   BUS_SELECT_REG            (Ulong)0x0020
#define   BUS_MON_SETUP_PTR         (Ulong)0x0022
#define   BUS_MON_CAR_HI            (Ulong)0x0024
#define   BUS_MON_CAR_LO            (Ulong)0x0026
#define   BUS_MON_TOR_HI            (Ulong)0x0028
#define   BUS_MON_TOR_LO            (Ulong)0x002A
#define   LOAD_CLK_HI_REG           (Ulong)0x002C
#define   LOAD_CLK_LO_REG           (Ulong)0x002E
#define   INTER_WORD_TIMEOUT        (Ulong)0x0030
#define   WATCHDOG_TIMEOUT          (Ulong)0x0032
#define   WATCHDOG_CHANGE           (Ulong)0x0034
#define   POLLING_TIMEOUT           (Ulong)0x0036
#define   TX_AMPLITUDE_REG          (Ulong)0x0038
#define   UPLINK_Q_PTR              (Ulong)0x003A
#define   DNLINK_Q_PTR              (Ulong)0x003C
#define   TX_PRI_REG                (Ulong)0x003E
#define   TX_SEC_REG                (Ulong)0x0040
#define   RX_PRI_REG                (Ulong)0x0042
#define   RX_SEC_REG                (Ulong)0x0044
#define   LDB_RT_ADDRESS_REG        (Ulong)0x0046
#define   WRAP_ROUND_HI_REG         (Ulong)0x0048
#define   WRAP_ROUND_LO_REG         (Ulong)0x004A

#define   SERIAL_NO_HI_REG          (Ulong)0x03F6
#define   SERIAL_NO_LO_REG          (Ulong)0x03F8
#define   FIRMWARE_CHECKSUM_HI      (Ulong)0x03FA
#define   FIRMWARE_CHECKSUM_LO      (Ulong)0x03FC
#define   FIRMWARE_REV_LEVEL        (Ulong)0x03FE

#define   BASE_REG_SZ               (Ulong)0x0400

/*
 Queue, message, data and SIM table pointers
*/
#define   LDB_PROGRAM               (Ulong)BASE_REG_SZ
#define   LDB_PROGRAM_SZ            (Ulong)0x0800

#define   BIN_Q                     (Ulong)(LDB_PROGRAM+LDB_PROGRAM_SZ)
#define   BIN_Q_SZ                  (Ulong)0x0200
#define   BIN_Q_ELEMENT_SZ          (Ulong)16
#define   HI_PRI_Q                  (Ulong)(BIN_Q+BIN_Q_SZ)
#define   HI_PRI_Q_SZ               (Ulong)0x0200
#define   LO_PRI_Q                  (Ulong)HI_PRI_Q+HI_PRI_Q_SZ
#define   LO_PRI_Q_SZ               (Ulong)0x0200
#define   LDB_Q_SZ                  (Ulong)0x0200
#define   LDB_BCT_Q_ELEMENT_SZ      (Ulong)4
#define   LDB_MRT_Q_ELEMENT_SZ      (Ulong)8


#define   RT_SIM_TABLE              (Ulong)(LO_PRI_Q+LO_PRI_Q_SZ)
#define   RT_SZ                     (Ulong)0x40
#define   RT_COUNT                  (Ulong)0x0020

#define   MON_SETUP                 (Ulong)(RT_SIM_TABLE+RT_SZ)
#define   MON_BIT_MODE              (Ulong)MON_SETUP+0
#define   MON_BUS_DEFINITION        (Ulong)MON_SETUP+2
#define   MON_BUS_DEF_DISABLE       (Ulong)MON_SETUP+4
#define   MON_PATTERN_HGH           (Ulong)MON_SETUP+6
#define   MON_PATTERN_MID           (Ulong)MON_SETUP+8
#define   MON_PATTERN_LOW           (Ulong)MON_SETUP+10
#define   MON_MASK_HGH              (Ulong)MON_SETUP+12
#define   MON_MASK_MID              (Ulong)MON_SETUP+14
#define   MON_MASK_LOW              (Ulong)MON_SETUP+16
#define   MON_ERROR_EVENT           (Ulong)MON_SETUP+18
#define   MON_SELECT_LUPT_PTR       (Ulong)MON_SETUP+20
#define   MON_PTC_HGH               (Ulong)MON_SETUP+22
#define   MON_PTC_LOW               (Ulong)MON_SETUP+24
#define   MON_STACK_START_HI        (Ulong)MON_SETUP+26
#define   MON_STACK_START_LO        (Ulong)MON_SETUP+28
#define   MON_STACK_STOP_HI         (Ulong)MON_SETUP+30
#define   MON_STACK_STOP_LO         (Ulong)MON_SETUP+32
#define   MON_RT_SELECT_TABLE       (Ulong)MON_SETUP+64
#define   MON_RT_SELECT_TABLE_SZ    (Ulong)0x0200
#define   MON_SETUP_SZ              (Ulong)0x0240

#define   MDB_START                 (Ulong)(MON_SETUP+MON_SETUP_SZ)
#define   MDB_SZ                    (Ulong)0x0020
#define   MDB_COUNT                 (Ulong)99
#define   MSG_COUNT                 (Uword)MDB_COUNT
#define   MDB_TOTAL_SZ              (Ulong)MDB_COUNT*MDB_SZ

#define   MON_RESERVED              (Ulong)0x10000
#define   MON_RESERVED_SZ           (Ulong)0x10000

#define   MON_STACK_SZ              (Ulong)0x10000
#define   MON_STACK_START_ADDR      (Ulong)(MON_RESERVED+MON_RESERVED_SZ)
#define   MON_STACK_STOP_ADDR       (Ulong)(MON_STACK_START_ADDR+MON_STACK_SZ)

#define   DATA_BLOCKS               (Ulong)MON_STACK_STOP_ADDR
#define   DATA_BLOCK_SZ             (Ulong)0x1020
#define   DATA_BLOCK_COUNT          (Ulong)MDB_COUNT
#define   MAX_DATA_SZ               (Ulong)1024
#define   TOTAL_DATA_SZ             (Ulong)(DATA_BLOCKS*DATA_BLOCK_SZ)

/****************************************************************************
              General definitions, structures and prototypes
****************************************************************************/

/**************************** Definitions **********************************/

#define   J1_CONNECTOR              (Uword)0x0008
#define   J2_CONNECTOR              (Uword)0x0004
#define   J3_CONNECTOR              (Uword)0x0002
#define   J4_CONNECTOR              (Uword)0x0001
#define   ANY_CONNECTOR             (Uword)0x8000

#define   EXECUTE_COMMAND           (Uword)0x0302
#define   PCI_CLOCK_READ            (Uword)0x0303

/*
 Card mode, command and status definitions.
*/
#define   COMMAND_REQ               (Uword)0x0302

#define   LDB_BCT_MODE              (Uword)0x0001
#define   LDB_MRT_MODE              (Uword)0x0002
#define   BUS_MON_MODE              (Uword)0x0003
#define   GDB_BCT_MODE              (Uword)0x0004
#define   GDB_MRT_MODE              (Uword)0x0005
#define   MDM_MRT_MODE              (Uword)0x0006
#define   PRM_MDE_MODE              (Uword)0x0007

#define   GOTO_LDB_BCT              (Uword)0x0001
#define   GOTO_LDB_MRT              (Uword)0x0002
#define   GOTO_BUS_MON              (Uword)0x0003
#define   GOTO_GDB_BCT              (Uword)0x0004
#define   GOTO_GDB_MRT              (Uword)0x0005
#define   GOTO_MDM_MRT              (Uword)0x0006
#define   GOTO_PRM_MDE              (Uword)0x0007
#define   START_CARD                (Uword)0x000A
#define   STOP_CARD                 (Uword)0x000B
#define   LOAD_CLOCK                (Uword)0x000C
#define   EXECUTE_SELFTEST          (Uword)0x000D
#define   EXECUTE_WRAP_TEST         (Uword)0x000E

#define   LDB_BCT_IDLE              (Uword)0x0010
#define   LDB_MRT_IDLE              (Uword)0x0020
#define   BUS_MON_IDLE              (Uword)0x0030
#define   GDB_BCT_IDLE              (Uword)0x0040
#define   GDB_MRT_IDLE              (Uword)0x0050
#define   MDM_MRT_IDLE              (Uword)0x0060
#define   PRM_MDE_IDLE              (Uword)0x0070
#define   LDB_BCT_RUNNING           (Uword)0x0011
#define   LDB_BCT_SW_PAUSED         (Uword)0x9011
#define   LDB_BCT_HW_PAUSED         (Uword)0xA011
#define   LDB_MRT_RUNNING           (Uword)0x0021
#define   BUS_MON_RUNNING           (Uword)0x0031
#define   GDB_BCT_RUNNING_ACTIVE    (Uword)0x0041
#define   GDB_BCT_RUNNING_PASSIVE   (Uword)0x9041
#define   GDB_BCT_HW_PAUSED         (Uword)0xA041
#define   GDB_MRT_RUNNING           (Uword)0x0051
#define   MDM_MRT_RUNNING           (Uword)0x0061
#define   PRM_MDE_RUNNING           (Uword)0x0071
#define   EXECUTING_SELFTEST        (Uword)0x000F
#define   SELFTEST_COMPLETE         (Uword)0x800F
#define   EXECUTING_WRAPTEST        (Uword)0x000A
#define   WRAPTEST_COMPLETE         (Uword)0x800A
#define   WRAPTEST_INTERNAL_PASS    (Uword)0x80FA

/*
 Base register default values.
*/
#define   BIN_Q_EVENT_DEF           (Uword)0x00FF
#define   VME_IRQ_1_DEF             (Uword)0x1980
#define   VME_IRQ_2_DEF             (Uword)0x1A81
#define   IRQ_SELECTION_DEF         (Uword)0x0000
#define   GAP_TIME_REG_DEF          (Uword)0x000B
#define   RT_RESPONSE_TIME_DEF      (Uword)15
#define   RT_RESPONSE_TIMEOUT_DEF   (Uword)50
#define   BUS_SELECT_REG_DEF        (Uword)0x0203
#define   INTER_WORD_TIMEOUT_DEF    (Uword)10
#define   WATCHDOG_TIMEOUT_DEF      (Uword)0x8010
#define   POLLING_TIMEOUT_DEF       (Uword)0x8010
#define   TX_AMPLITUDE_DEF          (Uword)20
#define   TX_PRI_REG_DEF            J1_CONNECTOR
#define   TX_SEC_REG_DEF            J3_CONNECTOR
#define   RX_PRI_REG_DEF            J2_CONNECTOR
#define   RX_SEC_REG_DEF            J4_CONNECTOR
#define   LDB_MRT_RT_DEF            (Uword)1
#define   WRAP_ROUND_HI_REG_DEF     (Uword)0x40AA
#define   WRAP_ROUND_LO_REG_DEF     (Uword)0xAAAA

/*
 Bus Monitor setup default values.
*/
#define   MON_BIT_MODE_DEF          (Uword)0x0000
#define   MON_BUS_DEFINITION_DEF    (Uword)0x0000
#define   MON_BUS_DEF_DISABLE_DEF   (Uword)0xC000
#define   MON_PATTERN_HGH_DEF       (Uword)0x0000
#define   MON_PATTERN_MID_DEF       (Uword)0x0000
#define   MON_PATTERN_LOW_DEF       (Uword)0x0000
#define   MON_MASK_HGH_DEF          (Uword)0x00FF
#define   MON_MASK_MID_DEF          (Uword)0xFFFF
#define   MON_MASK_LOW_DEF          (Uword)0xFFFF
#define   MON_ERROR_EVENT_DEF       (Uword)0x8030
#define   MON_SELECT_LUPT_PTR_DEF   (Uword)MON_RT_SELECT_TABLE
#define   MON_PTC_HGH_DEF           (Uword)0
#define   MON_PTC_LOW_DEF           (Uword)10

/*
 Setup array definitions.
*/
#define   F_SET_BITS_PER_WORD       (Ulong)0x00000001
#define   F_SET_GAP_TIME            (Ulong)0x00000002
#define   F_SET_RESPONSE_TIME       (Ulong)0x00000004
#define   F_SET_RESPONSE_TIMEOUT    (Ulong)0x00000008
#define   F_SET_BUS_SELECT          (Ulong)0x00000010
#define   F_SET_INTER_WORD_TIMEOUT  (Ulong)0x00000020
#define   F_SET_WATCHDOG_TIMEOUT    (Ulong)0x00000040
#define   F_SET_POLLING_TIMEOUT     (Ulong)0x00000080
#define   F_SET_TX_AMPLITUDE        (Ulong)0x00000100
#define   F_SET_VME_IRQ_1           (Ulong)0x00000200
#define   F_SET_VME_IRQ_2           (Ulong)0x00000400
#define   F_SET_IRQ_SELECTION       (Ulong)0x00000800
#define   F_SET_CONNECTORS          (Ulong)0x00001000
#define   F_SET_CHECKSUM_ON_OFF     (Ulong)0x00002000
#define   F_SET_GLOBAL_ERRS_ON_OFF  (Ulong)0x00004000
#define   F_SET_CLOCK_IRIG_FREE     (Ulong)0x00008000
#define   F_SET_ALL_PARAMETERS      (Ulong)0x0000FFFF

#define   SET_BITS_PER_WORD         0
#define   SET_GAP_TIME              1
#define   SET_RESPONSE_TIME         2
#define   SET_RESPONSE_TIMEOUT      3
#define   SET_BUS_SELECT            4
#define   SET_INTER_WORD_TIMEOUT    5
#define   SET_WATCHDOG_TIMEOUT      6
#define   SET_POLLING_TIMEOUT       7
#define   SET_TX_AMPLITUDE          8
#define   SET_VME_IRQ_1             9
#define   SET_VME_IRQ_2             10
#define   SET_IRQ_SELECTION         11
#define   SET_TX_PRI_CONNECTOR      12
#define   SET_TX_SEC_CONNECTOR      13
#define   SET_RX_PRI_CONNECTOR      14
#define   SET_RX_SEC_CONNECTOR      15
#define   SET_CHECKSUM_ON_OFF       16
#define   SET_GLOBAL_ERRS_ON_OFF    17
#define   SET_CLOCK_IRIG_FREE       18
#define   SETUP_INFO_SZ             19

#define   DATA_BITS_24              (Uword)0
#define   DATA_BITS_40              (Uword)0x8000
#define   CHECKSUM_ON               (Uword)0
#define   CHECKSUM_OFF              (Uword)0x0200
#define   GLOBAL_ERRS_ON            (Uword)0
#define   GLOBAL_ERRS_OFF           (Uword)0x0100
#define   CLOCK_FREE_RUNNING        (Uword)0
#define   CLOCK_DEMODULATE_IRIG     (Uword)0x8000
#define   GAP_TIME_MAX              (Uword)31

/*
 TX error definitions.
*/
#define   TX_ERROR_MASK             (Uword)0x01C0
#define   TX_NO_ERROR               (Uword)0x0100
#define   TX_PARITY_ERROR           (Uword)0x0000
#define   TX_SYNC_ERROR             (Uword)0x0040
#define   TX_MANCHESTER_ERROR       (Uword)0x0080
#define   TX_BIT_COUNT_ERROR        (Uword)0x00C0

/*
 LDB data buffer definitions.
*/
#define   LDB_DATA_TTAGHI           (Ulong)0
#define   LDB_DATA_TTAGMD           (Ulong)2
#define   LDB_DATA_TTAGLO           (Ulong)4
#define   LDB_DATA_TTAGTK           (Ulong)6
#define   LDB_DATA_BUFFSZ           (Ulong)8
#define   LDB_DATA_STATUS           (Ulong)10
#define   LDB_DATA_WORDCT           (Ulong)12
#define   LDB_DATA_CHKSUM           (Ulong)14
#define   LDB_DATA_NEXTHI           (Ulong)16
#define   LDB_DATA_NEXTLO           (Ulong)18
#define   LDB_DATA_FIRSTD           (Ulong)20

/*
 Queue definitions.
*/
#define   NO_Q_ELEMENT              (Uword)0
#define   GOT_Q_ELEMENT             (Uword)1
#define   CLEAR_Q                   (Uword)0
#define   READ_Q                    (Uword)1
#define   READ_Q_AND_CLEAR          (Uword)2
#define   HI_PRIORITY_Q             (Uword)1
#define   LO_PRIORITY_Q             (Uword)0

/*
 LDB BIN Queue definitions.
*/
#define   BIN_SY_Q_ERR              (Uword)0x1000
#define   BIN_PY_Q_ERR              (Uword)0x0800
#define   BIN_MN_Q_ERR              (Uword)0x0400
#define   BIN_LG_Q_ERR              (Uword)0x0200
#define   BIN_SH_Q_ERR              (Uword)0x0100

#define   BIN_TYPE_MASK             (Uword)0xC000
#define   BIN_UNSOL_TX_PRI          (Uword)0x0000
#define   BIN_UNSOL_TX_SEC          (Uword)0x4000
#define   BIN_UNSOL_RX_PRI          (Uword)0x8000
#define   BIN_UNSOL_RX_SEC          (Uword)0xC000

#define   BIN_ALT_GOOD_CMD          (Uword)0x8000
#define   BIN_ALT_GOOD_DAT          (Uword)0x4000
#define   BIN_ALT_BAD_CMD           (Uword)0x2000
#define   BIN_ALT_BAD_DAT           (Uword)0x1000
#define   BIN_ALT_PY_ERR            (Uword)0x0800
#define   BIN_ALT_MN_ERR            (Uword)0x0400
#define   BIN_ALT_LG_ERR            (Uword)0x0200
#define   BIN_ALT_SH_ERR            (Uword)0x0100

/*
 LDB_BCT message Queue definitions.
*/
#define   BCT_BS_Q_ERR              (Uword)0x0400
#define   BCT_OV_Q_ERR              (Uword)0x0200
#define   BCT_ID_Q_ERR              (Uword)0x0100
#define   BCT_NR_Q_ERR              (Uword)0x0080
#define   BCT_TA_Q_ERR              (Uword)0x0040
#define   BCT_WC_Q_ERR              (Uword)0x0020
#define   BCT_SY_Q_ERR              (Uword)0x0010
#define   BCT_PY_Q_ERR              (Uword)0x0008
#define   BCT_MN_Q_ERR              (Uword)0x0004
#define   BCT_LG_Q_ERR              (Uword)0x0002
#define   BCT_SH_Q_ERR              (Uword)0x0001

/*
 LDB_MRT message Queue definitions.
*/
#define   MRT_PT_Q_ERR              (Uword)0x8000
#define   MRT_RC_Q_ERR              (Uword)0x4000
#define   MRT_IS_Q_ERR              (Uword)0x2000
#define   MRT_CK_Q_ERR              (Uword)0x1000
#define   MRT_IG_Q_ERR              (Uword)0x0800
#define   MRT_BH_Q_ERR              (Uword)0x0400
#define   MRT_OV_Q_ERR              (Uword)0x0200
#define   MRT_ID_Q_ERR              (Uword)0x0100
#define   MRT_NR_Q_ERR              (Uword)0x0080
#define   MRT_TA_Q_ERR              (Uword)0x0040
#define   MRT_WC_Q_ERR              (Uword)0x0020
#define   MRT_SY_Q_ERR              (Uword)0x0010
#define   MRT_PY_Q_ERR              (Uword)0x0008
#define   MRT_MN_Q_ERR              (Uword)0x0004
#define   MRT_LG_Q_ERR              (Uword)0x0002
#define   MRT_SH_Q_ERR              (Uword)0x0001

/**************************** Structures ***********************************/

typedef struct
{
    Uword  mode;                                 /* Operating mode         */
    Uword  *vmeAddress;                          /* VME address pointer    */
    Ulong  pciAddress;                           /* PCI address pointer    */
    Uword  ldbMrtRt;                             /* LDB MRT number         */
    #if(SBIIB_PCI_PLATFORM)
        HANDLE            hDRVR;                 /* WinDrvr variables      */
        WD_CARD_REGISTER  cardReg;
        WD_VERSION        verBuf;
        WD_LICENSE        lic;
        WD_PCI_SCAN_CARDS pciScan;
        WD_PCI_CARD_INFO  pciCardInfo;
        WD_TRANSFER       trans;
        DWORD             G_resMemory;
        DWORD             G_resInterrupt;
        DWORD             dCntr;
    #endif
} CardHandle;

typedef struct
{
    Ulong    serialNo;                           /* Card serial number     */
    Uword    checksumH;                          /* Firmware checksum HI   */
    Uword    checksumL;                          /* Firmware checksum LO   */
    Uword    fwRev;                              /* Card FW Revision Level */
    Sbyte    drvsRev[SBIIB_DRVS_VERSION_SIZE];   /* Drivers Revision Level */
} CardID;
 
typedef struct
{
    Ulong    whatToSet;                          /* What to set            */
    Uword    settings[SETUP_INFO_SZ];            /* Setup values array     */
} SetupInfo;

typedef struct
{
    Uword    days;                               /* Days                   */
    Uword    hours;                              /* Hours                  */
    Uword    minutes;                            /* Minutes                */
    Uword    seconds;                            /* Seconds                */
    Uword    msecs;                              /* mS                     */
    Uword    micros;                             /* uS                     */
    Uword    halfuS;                             /* half uS flag           */
    Uword    lock;                               /* IRIG-B lock flag       */
    Uword    ttagH;                              /* Clock value HGH        */
    Uword    ttagM;                              /* Clock value MID        */
    Uword    ttagL;                              /* Clock value LOW        */
    Uword    ttagT;                              /* Clock value TCK        */
} TimeTag;

typedef struct
{
    Uword    wordNo;                             /* Error word number      */
    Uword    errType;                            /* Error type             */
    Uword    errInfo;                            /* Error info             */
} TXerrs;

typedef struct
{
    TimeTag  time;                               /* TimeTag structure      */
    Uword    buffSZ;                             /* Buffer overflow size   */
    Uword    status;                             /* Buffer status          */
    Uword    wCount;                             /* Buffer RX or TX count  */
    Uword    chksum;                             /* Buffer RT checksum     */
    Ulong    *data;                              /* Pointer to data array  */
} DBuff;

typedef struct
{
    Uword    mode;                               /* Mode - free or IRIG-B  */
    Uword    leapYear;                           /* leap year value 0-3    */
    Uword    days;                               /* Days 0-366             */
    Uword    hours;                              /* Hours 0-23             */
    Uword    minutes;                            /* Minutes 0-59           */
    Uword    seconds;                            /* Seconds 0-59           */
} ClockInit;

typedef struct
{
    Uword    action;                             /* Q action               */
    Uword    result;                             /* Result of action       */
    Ulong    offset;                             /* Address pointer in Q   */
    TimeTag  time;                               /* Time tag storage       */
    Ulong    data;                               /* Data word              */
    Uword    altData;                            /* Alternate bus data     */
} LDB_BIN_Qdata;

/************************ SB_IOFNS.C Prototypes ****************************/

Error  SBIIB_Init       (CardHandle *card, Ulong cardAddress);
Error  SBIIB_RdClock    (CardHandle *card, TimeTag *time);
void   wW               (CardHandle *card, Ulong offset, Uword value);
Uword  rW               (CardHandle *card, Ulong offset);

/************************ SB_SETUP.C Prototypes ****************************/

Error  SBIIB_Setup      (CardHandle *card, Uword mode, SetupInfo *info);
Error  SBIIB_RdSetup    (CardHandle *card, SetupInfo *info);
Error  SBIIB_SetupLdbBc (CardHandle *card);
Error  SBIIB_SetupLdbRt (CardHandle *card);
Error  SBIIB_SetupMon   (CardHandle *card);
Error  SBIIB_SetupMdmRt (CardHandle *card);

/************************ SB_LIBRY.C Prototypes ****************************/

Error  SBIIB_RdStatus   (CardHandle *card, Uword  *status);
Error  SBIIB_GetCardID  (CardHandle *card, CardID *info);
Error  SBIIB_Command    (CardHandle *card, Uword command);
Error  SBIIB_SetErr     (CardHandle *card, Ulong offset, TXerrs *info);
Error  SBIIB_GetErr     (CardHandle *card, Ulong offset, TXerrs *info);
Error  SBIIB_RdTtag     (CardHandle *card, Ulong offset, TimeTag *time);
Error  SBIIB_LoadClock  (CardHandle *card, ClockInit *info);
Error  SBIIB_FormatTtag (Uword hgh, Uword mid,
                         Uword low, Uword tck, TimeTag *time);
Error  SBIIB_RdBinQ     (CardHandle *card, LDB_BIN_Qdata *qData);

/****************************************************************************
            LDB_BCT specific definitions, structures and prototypes
****************************************************************************/

/**************************** Definitions **********************************/

#define   LDB_MDB_NUMBER            (Ulong)0
#define   LDB_MDB_IRQCON            (Ulong)2
#define   LDB_MDB_CMNDHI            (Ulong)4
#define   LDB_MDB_CMNDLO            (Ulong)6
#define   LDB_MDB_TXERRS            (Ulong)8
#define   LDB_MDB_TXERWD            (Ulong)10
#define   LDB_MDB_DATAHI            (Ulong)12
#define   LDB_MDB_DATALO            (Ulong)14
#define   LDB_MDB_DATAWC            (Ulong)16
#define   LDB_MDB_MEVENT            (Ulong)18
#define   LDB_MDB_PATTHI            (Ulong)20
#define   LDB_MDB_PATTLO            (Ulong)22
#define   LDB_MDB_MASKHI            (Ulong)24
#define   LDB_MDB_MASKLO            (Ulong)26
#define   LDB_MDB_GAPTHI            (Ulong)28
#define   LDB_MDB_GAPTLO            (Ulong)30

#define   LDB_DELAY                 (Uword)0x0000
#define   LDB_NOP1                  (Uword)0x0001
#define   LDB_NOP2                  (Uword)0x0002
#define   LDB_NOP3                  (Uword)0x0003
#define   LDB_BSR                   (Uword)0x0004
#define   LDB_JSR                   (Uword)0x0005
#define   LDB_BRA                   (Uword)0x0006
#define   LDB_JMP                   (Uword)0x0007
#define   LDB_RTS                   (Uword)0x0008
#define   LDB_LOOP                  (Uword)0x0009
#define   LDB_DBNE                  (Uword)0x000A
#define   LDB_INITF                 (Uword)0x000B
#define   LDB_SWPSE                 (Uword)0x000C
#define   LDB_HALT                  (Uword)0x000D
#define   LDB_TRIGOUT               (Uword)0x000E
#define   LDB_SMB                   (Uword)0x000F
#define   LDB_SITH                  (Uword)0x0010
#define   LDB_SITL                  (Uword)0x0011
#define   LDB_HWPSE                 (Uword)0x0012

#define   LDB_TX_FOREVER            (Uword)0x0000

#define   PRIMARY_BUS               (Uword)0x8000
#define   SECONDARY_BUS             (Uword)0x4000

#define   RX_MESSAGE                (Uword)0x0000
#define   TX_MESSAGE                (Uword)0x0001

#define   SEV_TEST                  (Uword)0x0000
#define   SEV_NO_TEST               (Uword)0x0001

/**************************** Structures ***********************************/

typedef struct
{
    Ulong    fTime;                              /* Frame duration         */
    Uword    fCount;                             /* Frame TX count         */
    Uword    mCount;                             /* Frame message count    */
    Uword    *msgs;                              /* Pointer to message Nos */
} LDB_BCT_PROGinfo;

typedef struct
{
    Uword    msgNo;                              /* Message number         */
    Uword    msgType;                            /* TX or RX message type  */
    Uword    dataCount;                          /* message data count     */
    Uword    sevTest;                            /* SEV test flag          */
    Uword    busID;                              /* TX bus ID              */
    Ulong    command;                            /* Command word           */
    TXerrs   *txErrs;                            /* TX error injection     */
    Ulong    gapTime;                            /* Intermessage gap time  */
    Uword    irqDef;                             /* IRQ defiition word     */
    DBuff    *data;                              /* Pointer to data        */
} LDB_BCT_MSGinfo;

typedef struct
{
    Uword    action;                             /* Q action               */
    Uword    result;                             /* Result of action       */
    Uword    priority;                           /* HI or LO queue         */
    Ulong    offset;                             /* Address pointer in Q   */
    Uword    messNo;                             /* Message number         */
    Uword    errors;                             /* Q error word           */
} LDB_BCT_Qdata;

/**************************** Prototypes ***********************************/

Error  SBIIB_WrProg   (CardHandle *card, LDB_BCT_PROGinfo *info);
Error  SBIIB_RdProg   (CardHandle *card, LDB_BCT_PROGinfo *info);
Error  SBIIB_WrMsg    (CardHandle *card, LDB_BCT_MSGinfo  *info);
Error  SBIIB_RdMsg    (CardHandle *card, LDB_BCT_MSGinfo  *info);
Error  SBIIB_RdLdbBcQ (CardHandle *card, LDB_BCT_Qdata    *qData);
Error  SBIIB_RunLdbBc (CardHandle *card, Uword count);

/****************************************************************************
            LDB_MRT specific definitions, structures and prototypes
****************************************************************************/

/**************************** Definitions **********************************/

#define   LDB_SIM_MODEWD            (Ulong)0
#define   LDB_SIM_STATEM            (Ulong)2
#define   LDB_SIM_IRQWDA            (Ulong)4
#define   LDB_SIM_IRQWDB            (Ulong)6
#define   LDB_SIM_TXBFHI            (Ulong)8
#define   LDB_SIM_TXBFLO            (Ulong)10
#define   LDB_SIM_RXBFHI            (Ulong)12
#define   LDB_SIM_RXBFLO            (Ulong)14
#define   LDB_SIM_TTAGHI            (Ulong)16
#define   LDB_SIM_TTAGMD            (Ulong)18
#define   LDB_SIM_TTAGLO            (Ulong)20
#define   LDB_SIM_TTAGTK            (Ulong)22
#define   LDB_SIM_LCMDHI            (Ulong)24
#define   LDB_SIM_LCMDLO            (Ulong)26
#define   LDB_SIM_LRESHI            (Ulong)28
#define   LDB_SIM_LRESLO            (Ulong)30
#define   LDB_SIM_GAHERR            (Ulong)32
#define   LDB_SIM_GAHWRD            (Ulong)34
#define   LDB_SIM_INDERR            (Ulong)36
#define   LDB_SIM_INDWRD            (Ulong)38
#define   LDB_SIM_IWDERR            (Ulong)40
#define   LDB_SIM_IWDWRD            (Ulong)42
#define   LDB_SIM_SRQERR            (Ulong)44
#define   LDB_SIM_SRQWRD            (Ulong)46
#define   LDB_SIM_FRCERR            (Ulong)48
#define   LDB_SIM_FRCWRD            (Ulong)50
#define   LDB_SIM_RESVD1            (Ulong)52
#define   LDB_SIM_RESVD2            (Ulong)54
#define   LDB_SIM_RESVD3            (Ulong)56
#define   LDB_SIM_RESVD4            (Ulong)58
#define   LDB_SIM_RESVD5            (Ulong)60
#define   LDB_SIM_RESVD6            (Ulong)62

#define   LDB_MRT_MODE_MASK         (Uword)0x8000
#define   LDB_MRT_ACTIVE            (Uword)0x8000
#define   LDB_MRT_PASSIVE           (Uword)0x0000
#define   LDB_MRT_DDT_MASK          (Uword)0x03FF

#define   SRQ                       (Uword)0x0002
#define   IND                       (Uword)0x0003
#define   GAH                       (Uword)0x0005
#define   HCD                       (Uword)0x0009
#define   STA                       (Uword)0x000A
#define   IWD                       (Uword)0x000D

#define   NNB                       (Uword)0x0003
#define   DTT                       (Uword)0x0006
#define   GTX                       (Uword)0x000C
#define   TXD                       (Uword)0x000B

#define   WFI                       (Uword)0x0001
#define   WFGAH                     (Uword)0x0002
#define   WFHCD                     (Uword)0x0004
#define   WFS                       (Uword)0x0008
#define   WFSR                      (Uword)0x0010
#define   WFSRE                     (Uword)0x0020
#define   FORCED                    (Uword)0x0040

#define   UPDATE_STATE_OFF          (Uword)0x0000
#define   UPDATE_STATE_ON           (Uword)0x0001

#define   NO_DATA_AND_READY         (Uword)0x0000
#define   NO_DATA_NOT_READY         (Uword)0x1000
#define   CRITICAL_DATA             (Uword)0x0800
#define   NON_CRITICAL_AND_READY    (Uword)0x0C00
#define   NON_CRITICAL_NOT_READY    (Uword)0x1C00
#define   LDB_MRT_ACTION_MASK       (Uword)0x1C00
#define   LDB_MRT_ACTION_CLEAR      (Uword)0xE3FF

#define   LDB_MRT_SELECT_PRI        (Uword)0x0002
#define   LDB_MRT_SELECT_SEC        (Uword)0x0001
#define   LDB_MRT_SELECT_BOTH       (Uword)0x0003

#define   NO_WRAP                   (Uword)0
#define   WRAP                      (Uword)1

/**************************** Structures ***********************************/

typedef struct
{
    Uword    rtNo;                               /* RT number              */
    Uword    bus;                                /* Bus select             */
    Uword    mode;                               /* RT mode                */
    Uword    action;                             /* Critical etc.          */
    Uword    ddtCount;                           /* DDT wordcount          */
    Uword    state;                              /* RT state machine       */
    Uword    updateState;                        /* RT state update flag   */
    Uword    irqDefA;                            /* IRQ definition A       */
    Uword    irqDefB;                            /* IRQ definition B       */
    Uword    wrap;                               /* Data wrap mode         */
    TimeTag  time;                               /* TimeTag structure      */
    Ulong    lastCmnd;                           /* Last command           */
    Ulong    lastResp;                           /* Last response          */
    TXerrs   *GAHerrs;                           /* GAH TX error injection */
    TXerrs   *INDerrs;                           /* IND TX error injection */
    TXerrs   *IWDerrs;                           /* IWD TX error injection */
    TXerrs   *SRQerrs;                           /* SRQ TX error injection */
    TXerrs   *FRCerrs;                           /* FRC TX error injection */
    DBuff    *txData;                            /* Pointer to TX data     */
    DBuff    *rxData;                            /* Pointer to RX data     */
} LDB_MRTinfo;

typedef struct
{
    Uword    action;                             /* Q action               */
    Uword    result;                             /* Result of action       */
    Uword    priority;                           /* HI or LO queue         */
    Ulong    offset;                             /* Address pointer in Q   */
    Ulong    command;                            /* Q command word         */
    Uword    errors;                             /* Q error word           */
	 Uword    altData;                            /* Alternate bus data     */
} LDB_MRT_Qdata;

/**************************** Prototypes ***********************************/

Error  SBIIB_WrLdbRT       (CardHandle *card, LDB_MRTinfo *info);
Error  SBIIB_RdLdbRT       (CardHandle *card, LDB_MRTinfo *info);
Error  SBIIB_RdLdbRTstate  (CardHandle *card, Uword *state);
Error  SBIIB_WrLdbRTaction (CardHandle *card, Uword action);
Error  SBIIB_RdLdbRtQ      (CardHandle *card, LDB_MRT_Qdata *qData);
Error  SBIIB_RunLdbRt      (CardHandle *card, Uword rtNo);

/****************************************************************************
            BUS_MON specific definitions, structures and prototypes
****************************************************************************/

/**************************** Definitions **********************************/

#define   MON_STACK_ELEMENT_SZ      16

#define   MON_RT_SELECTIVE_ON       (Uword)1
#define   MON_RT_SELECTIVE_OFF      (Uword)0

#define   MON_LDB_MODE              (Uword)0
#define   MON_GDB_MODE              (Uword)1

#define   MON_CMND_SYNC             (Uword)0x0020
#define   MON_DATA_SYNC             (Uword)0x0010
#define   MON_SYNC_TYPE             (Uword)0x0010
#define   MON_PARITY_ERROR          (Uword)0x0008
#define   MON_MANCHESTER_ERROR      (Uword)0x0004
#define   MON_LONGWORD_ERROR        (Uword)0x0002
#define   MON_SHORTWORD_ERROR       (Uword)0x0001
#define   MON_DISABLE_ERRORS        (Uword)0x8000

#define   MON_J1                    (Uword)0x0000
#define   MON_J2                    (Uword)0x4000
#define   MON_J3                    (Uword)0x8000
#define   MON_J4                    (Uword)0xC000

/**************************** Structures ***********************************/

typedef struct
{
    Uword    mode;                               /* LDB or GDB mode        */
    Uword    busDef;                             /* Bus definition         */
    Sbyte    pattern[40];                        /* Bit pattern (0,1,X)    */
    Uword    errorEvent;                         /* Errro event            */
    Ulong    ptc;                                /* Post trigger count     */
    Uword    *select;                            /* Selective table values */ 
} BUS_MON_TRIGinfo;

typedef struct
{
    TimeTag  time;                               /* Message TimeTag        */
    Uword    busID;                              /* Bus of RX message      */
    Uword    dataH;                              /* Data pattern HGH       */
    Uword    dataM;                              /* Data pattern MID       */
    Uword    dataL;                              /* Data pattern LOW       */
    Uword    errors;                             /* Error word             */
} BUS_MON_Msg;

/**************************** Prototypes ***********************************/

Error  SBIIB_WrMonTrigger(CardHandle *card, BUS_MON_TRIGinfo *info);
Error  SBIIB_RdMonTrigger(CardHandle *card, BUS_MON_TRIGinfo *info);
Error  SBIIB_RdMonMsg    (CardHandle *card, Slong msgNo, BUS_MON_Msg *info);
Error  SBIIB_RdMonMsgAbs (CardHandle *card, Ulong address, BUS_MON_Msg *info);
Error  SBIIB_RdMonCarTor (CardHandle *card, Ulong *car, Ulong *tor);
Error  SBIIB_ClrMonStack (CardHandle *card);
Error  SBIIB_RunMon      (CardHandle *card);

