/****************************************************************************

   Copyright (c) 2000, Western Avionics LTD.             All Rights Reserved.

*****************************************************************************

   File          : sb_bsmon.c

   Author        : S.J.Buckley

   Date          : 11-01-2000

   Function      : SBIIB BUS_MON functions.

   Update        : dd-mm-yy
   By            : S.J.Buckley
   Reason        :

****************************************************************************/
#include "sbiib.h"

/****************************************************************************

 Function  : SBIIB_WrMonTrigger

 Purpose   : This function sets the trigger for the bus monitor

 Input(s)  : card       - handle for the card
             info       - BUS_MON_TRIGinfo information structure

 Output(s) : error      - Function status

****************************************************************************/
Error  SBIIB_WrMonTrigger(CardHandle        *card,  /* Card handle         */
                          BUS_MON_TRIGinfo  *info)  /* Trigger structure   */
/***************************************************************************/
{
Error  error;
Ulong  aVal;
Uword  i, bdef, bddis;
Uword  pH, pM, pL, mH, mM, mL;
Uword  pBit, mBit;

    /*
     Initialise the variables.
    */
    error = E_NO_ERROR;
    pH    = 0;
    pM    = 0;
    pL    = 0;
    mH    = 0;
    mM    = 0;
    mL    = 0;
    bddis = 0;

    /*
     Test for valid pointers and parameters.
    */
    if(card == NO_POINTER || info == NO_POINTER)
        error = E_INVALID_POINTER;
    else
    if(card->mode != BUS_MON_MODE)
        error = E_INVALID_MODE;
    else
    if(info->mode != MON_LDB_MODE && info->mode != MON_GDB_MODE)
        error = E_INVALID_PARAMETER;
    else
    {
        if(info->busDef == J1_CONNECTOR)
            bdef = MON_J1;
        else
        if(info->busDef == J2_CONNECTOR)
            bdef = MON_J2;
        else
        if(info->busDef == J3_CONNECTOR)
            bdef = MON_J3;
        else
        if(info->busDef == J4_CONNECTOR)
            bdef = MON_J4;
        else
        if(info->busDef == ANY_CONNECTOR)
        {
            bdef  = MON_J4;
            bddis = MON_BUS_DEF_DISABLE_DEF;
        }
        else
            error = E_INVALID_PARAMETER;
    }

    /*
     Pass through the pattern string and form the 40 bit mask
     and 40 bit pattern. Set error if invalid character.
    */
    for(i=0; i!=40 && error==E_NO_ERROR; i++)
    {
        switch(info->pattern[i])
        {
            case '0':
                pBit = 0;
                mBit = 0;                        
                break;

            case '1':
                pBit = 0x8000;
                mBit = 0;                        
                break;

            case 'x':
            case 'X':
                pBit = 0;
                mBit = 0x8000;                        
                break;

            default:
                error = E_INVALID_PARAMETER;
                break;
        }

        if(i < 8)
        {
            pH |= (Uword)(pBit >> (i+8));
            mH |= (Uword)(mBit >> (i+8));
        }
        else
        if(i < 24)
        {
            pM |= (Uword)(pBit >> (i-8));
            mM |= (Uword)(mBit >> (i-8));
        }
        else
        {
            pL |= (Uword)(pBit >> (i-24));
            mL |= (Uword)(mBit >> (i-24));
        }
    }

    /*
     Save the parameters in the card registers.
    */
    if(error == E_NO_ERROR)
    {
        wW(card, MON_BIT_MODE, info->mode);
        wW(card, MON_BUS_DEFINITION, bdef);
        wW(card, MON_BUS_DEF_DISABLE, bddis);
        wW(card, MON_PATTERN_HGH, pH);
        wW(card, MON_PATTERN_MID, pM);
        wW(card, MON_PATTERN_LOW, pL);
        wW(card, MON_MASK_HGH, mH);
        wW(card, MON_MASK_MID, mM);
        wW(card, MON_MASK_LOW, mL);
        wW(card, MON_ERROR_EVENT, info->errorEvent);
        wW(card, MON_SELECT_LUPT_PTR, MON_SELECT_LUPT_PTR_DEF);
        wW(card, MON_PTC_HGH, (Uword)((info->ptc >> 16) & 0xFFFF));
        wW(card, MON_PTC_LOW, (Uword)(info->ptc & 0xFFFF));
    }

    /*
     If selective array required, copy the array into the table and check
     for valid values.
    */
    if(error == E_NO_ERROR && info->select != NO_POINTER)
    {
        for(aVal=0; aVal != MON_RT_SELECT_TABLE_SZ; aVal+=2)
        {
            bdef = info->select[aVal/2];
            wW(card, MON_RT_SELECT_TABLE+aVal, bdef);
            if(bdef != MON_RT_SELECTIVE_ON && bdef != MON_RT_SELECTIVE_OFF)
                error = E_INVALID_PARAMETER;
        }
    }

return (error);
}

/****************************************************************************

 Function  : SBIIB_RdMonTrigger

 Purpose   : This function reads back the trigger for the bus monitor

 Input(s)  : card       - handle for the card
             info       - BUS_MON_TRIGinfo information structure

 Output(s) : error      - Function status

****************************************************************************/
Error  SBIIB_RdMonTrigger(CardHandle        *card,   /* Card handle        */
                          BUS_MON_TRIGinfo  *info)   /* Trigger structure  */
/***************************************************************************/
{
Error  error;
Ulong  aVal;
Uword  i;
Uword  pH, pM, pL, mH, mM, mL;
Sbyte  bitV;

    /*
     Initialise the variables.
    */
    error = E_NO_ERROR;

    /*
     Test for valid pointers and parameters.
    */
    if(card == NO_POINTER || info == NO_POINTER)
        error = E_INVALID_POINTER;
    else
    if(card->mode != BUS_MON_MODE)
        error = E_INVALID_MODE;

    /*
     If good parmeters, save the trigger in the structure.
    */
    else
    {
        info->mode = rW(card, MON_BIT_MODE);
        i = rW(card, MON_BUS_DEF_DISABLE);

        if(i == MON_BUS_DEF_DISABLE_DEF)
            info->busDef = ANY_CONNECTOR;
        else
        {
            i = rW(card, MON_BUS_DEFINITION);

            if(i == MON_J1)
                info->busDef = J1_CONNECTOR;
            else
            if(i == MON_J2)
                info->busDef = J2_CONNECTOR;
            else
            if(i == MON_J3)
                info->busDef = J3_CONNECTOR;
            else
                info->busDef = J4_CONNECTOR;
        }

        /*
         Read the pattern.mask variables and covert into a 0,1,X string
         and save in the structure.
        */
        
        pH = rW(card, MON_PATTERN_HGH);
        pM = rW(card, MON_PATTERN_MID);
        pL = rW(card, MON_PATTERN_LOW);
        mH = rW(card, MON_MASK_HGH);
        mM = rW(card, MON_MASK_MID);
        mL = rW(card, MON_MASK_LOW);

        for(i=0; i!=40; i++)
        {
            if(i < 8)
            {
                if(mH & (Uword)(0x8000 >> (8+i)))
                    bitV = 'X';
                else
                if(pH & (Uword)(0x8000 >> (8+i)))
                    bitV = '1';
                else
                    bitV = '0';
            }
            else
            if(i < 24)
            {
                if(mM & (Uword)(0x8000 >> (i-8)))
                    bitV = 'X';
                else
                if(pM & (Uword)(0x8000 >> (i-8)))
                    bitV = '1';
                else
                    bitV = '0';
            }
            else
            {
                if(mL & (Uword)(0x8000 >> (i-24)))
                    bitV = 'X';
                else
                if(pL & (Uword)(0x8000 >> (i-24)))
                    bitV = '1';
                else
                    bitV = '0';
            }

            info->pattern[i] = bitV;
        }

        info->errorEvent = rW(card, MON_ERROR_EVENT);
        aVal  = (Ulong)rW(card, MON_PTC_HGH);
        aVal  = (aVal << 16) & 0xFFFF0000;
        aVal |= (Ulong)rW(card, MON_PTC_LOW) & 0xFFFF;
        info->ptc = aVal;
    }

    /*
     If selective array required, copy the array into the table and check
     for valid values.
    */

    if(error == E_NO_ERROR && info->select != NO_POINTER)
    {
        for(aVal=0; aVal != MON_RT_SELECT_TABLE_SZ; aVal+=2)
            info->select[aVal/2] = rW(card, MON_RT_SELECT_TABLE+aVal);
    }

return (error);
}

/****************************************************************************

 Function  : SBIIB_RdMonMsg

 Purpose   : This function reads a BUS_MON stack message at the given
             message number relative to the trigger.

 Input(s)  : card       - handle for the card
             msgNo      - Stack message number
             info       - BUS_MON_Msg information structure

 Output(s) : error      - Function status

****************************************************************************/
Error  SBIIB_RdMonMsg(CardHandle    *card,        /* Card handle           */
                      Slong         msgNo,        /* Signed message number */
                      BUS_MON_Msg   *info)        /* BUS_MON msg structure */
/***************************************************************************/
{
Error  error;
Ulong  tor, car, negVal, count, i;

    /*
     Read in the CAR and TOR values.
    */
    if(info == NO_POINTER)
        error = E_INVALID_POINTER;
    else
        error = SBIIB_RdMonCarTor(card, &car, &tor);

    /*
     Read the TOR and CAR from the base registers and test to ensure they
     are valid element addresses within the stack.
    */
    if(error == E_NO_ERROR)
    {
        if(tor == 0 || tor == car)
            error = E_TRIGGER_NOT_OCCURED;
        else
        if(tor > (MON_STACK_STOP_ADDR - MON_STACK_ELEMENT_SZ))
            error = E_INVALID_MON_TOR;
        else
        if(tor < MON_STACK_START_ADDR)
            error = E_INVALID_MON_TOR;
        else
        if((tor % MON_STACK_ELEMENT_SZ) != 0)
            error = E_INVALID_MON_TOR;
        else
        if(car > (MON_STACK_STOP_ADDR - MON_STACK_ELEMENT_SZ))
            error = E_INVALID_MON_CAR;
        else
        if(car < MON_STACK_START_ADDR)
            error = E_INVALID_MON_CAR;
        else
        if((car % MON_STACK_ELEMENT_SZ) != 0)
            error = E_INVALID_MON_CAR;
    }

    /*
     Now, based on th epolarity of the 'msgNo', find the absolute address
     and make sure that the message exists.
    */
    if(error == E_NO_ERROR)
    {

        if(msgNo >= 0)
        {
            count = (Ulong)msgNo;
            for(i=0; i < count; i++)
            {
                if(tor == car)
                    error = E_INVALID_MON_MESSAGE;
                else
                {
                    tor += MON_STACK_ELEMENT_SZ;
                    if(tor >= MON_STACK_STOP_ADDR)
                        tor = MON_STACK_START_ADDR;
                }
            }
        }
        else
        {
            count = (Ulong)(msgNo * -1);
            negVal = car - MON_STACK_ELEMENT_SZ;
            if(negVal < MON_STACK_START_ADDR)
                negVal = MON_STACK_STOP_ADDR - MON_STACK_ELEMENT_SZ;

            for(i=0; i < count; i++)
            {
                if(tor == negVal)
                    error = E_INVALID_MON_MESSAGE;
                else
                {
                    tor -= MON_STACK_ELEMENT_SZ;
                    if(tor < MON_STACK_START_ADDR)
                        tor = MON_STACK_STOP_ADDR - MON_STACK_ELEMENT_SZ;
                }
            }
        }
    }

    if(error == E_NO_ERROR)
        error = SBIIB_RdMonMsgAbs(card, tor, info);

return (error);
}

/****************************************************************************

 Function  : SBIIB_RdMonMsgAbs

 Purpose   : This function reads a BUS_MON stack message at the given
             absolute card address.

 Input(s)  : card       - handle for the card
             address    - Absolute card address of message
             info       - BUS_MON_Msg information structure

 Output(s) : error      - Function status

****************************************************************************/
Error  SBIIB_RdMonMsgAbs(CardHandle    *card,     /* Card handle           */
                         Ulong         address,   /* Message address       */
                         BUS_MON_Msg   *info)     /* BUS_MON msg structure */
/***************************************************************************/
{
Error  error;
Uword  pH, id;

    /*
     Test for valid pointers and parameters.
    */
    if(card == NO_POINTER || info == NO_POINTER)
        error = E_INVALID_POINTER;
    else
    if(card->mode != BUS_MON_MODE)
        error = E_INVALID_MODE;
    else
    if(address < MON_STACK_START_ADDR ||
       address > (MON_STACK_STOP_ADDR - MON_STACK_ELEMENT_SZ))
        error = E_INVALID_PARAMETER;
    else
    if((address % MON_STACK_ELEMENT_SZ) != 0)
        error = E_INVALID_PARAMETER;

    /*
     If good parameters, read in the stack values and format them into the
     structure.
    */
    else
    {
        pH = rW(card, address+8);
        id = pH & MON_J4;

        if(id == MON_J1)
            info->busID = J1_CONNECTOR;
        else
        if(id == MON_J2)
            info->busID = J2_CONNECTOR;
        else
        if(id == MON_J3)
            info->busID = J3_CONNECTOR;
        else
            info->busID = J4_CONNECTOR;

        info->errors = (Uword)((pH >> 8) & 0x001F);
        info->dataM  = rW(card, address+10);
        info->dataL  = rW(card, address+12);

        error = SBIIB_RdTtag(card, address, &info->time);
    }

return (error);
}

/****************************************************************************

 Function  : SBIIB_ClrMonStack

 Purpose   : Clears the BUS_MON stack.

 Input(s)  : card       - handle for the card

 Output(s) : error      - Function status

****************************************************************************/
Error  SBIIB_ClrMonStack(CardHandle  *card)       /* Card handle           */
/***************************************************************************/
{
Error  error;
Ulong  aVal;

    error = E_NO_ERROR;

    if(card == NO_POINTER)
        error = E_INVALID_POINTER;
    else
    if(card->mode != BUS_MON_MODE)
        error = E_INVALID_MODE;
    else
    {
        for(aVal=MON_STACK_START_ADDR; aVal!=MON_STACK_STOP_ADDR; aVal+=2)
            wW(card, aVal, 0);
    }

return (error);
}

/****************************************************************************

 Function  : SBIIB_RdMonCarTor

 Purpose   : Reads the current value of the BUS_MON CAR and TOR.

 Input(s)  : card       - handle for the card
             car        - CAR storage
             tor        - TOR storage

 Output(s) : error      - Function status

****************************************************************************/
Error  SBIIB_RdMonCarTor(CardHandle    *card,     /* Card handle           */
                         Ulong         *car,      /* CAR value             */
                         Ulong         *tor)      /* TOR value             */
/***************************************************************************/
{
Error  error;
Ulong  tV, cV;

    error = E_NO_ERROR;

    /*
     If parameters are valid, read in the CAR and TOT values.
    */
    if(card == NO_POINTER || car == NO_POINTER || tor == NO_POINTER)
        error = E_INVALID_POINTER;
    else
    if(card->mode != BUS_MON_MODE)
        error = E_INVALID_MODE;
    else
    {
        cV = (Ulong)rW(card, BUS_MON_CAR_HI);
        cV = (cV << 16) & 0xFFFF0000;
        *car = cV | ((Ulong)rW(card, BUS_MON_CAR_LO) & 0xFFFF);
        tV = (Ulong)rW(card, BUS_MON_TOR_HI);
        tV = (tV << 16) & 0xFFFF0000;
        *tor = tV | ((Ulong)rW(card, BUS_MON_TOR_LO) & 0xFFFF);
    }

return (error);
}

/****************************************************************************

 Function  : SBIIB_RunMon

 Purpose   : Runs the BUS_MON.

 Input(s)  : card       - handle for the card

 Output(s) : error      - Function status

****************************************************************************/
Error  SBIIB_RunMon(CardHandle    *card)          /* Card handle           */
/***************************************************************************/
{
Error  error;

    /*
     If parameters are valid, clear the CAR and TOR and run the BUS_MON.
    */
    if(card == NO_POINTER)
        error = E_INVALID_POINTER;
    else
    if(card->mode != BUS_MON_MODE)
        error = E_INVALID_MODE;
    else
    {
        wW(card, BUS_MON_CAR_HI, 0);
        wW(card, BUS_MON_CAR_LO, 0);
        wW(card, BUS_MON_TOR_HI, 0);
        wW(card, BUS_MON_TOR_LO, 0);
        error = SBIIB_Command(card, START_CARD);
    }

return (error);
}
