/****************************************************************************

   Copyright (c) 2000, Western Avionics LTD.             All Rights Reserved.

*****************************************************************************

   File          : sb_iofns.c

   Author        : S.J.Buckley

   Date          : 11-01-2000

   Function      : SBIIB card access module.

   Update        : dd-mm-yy
   By            : S.J.Buckley
   Reason        :

****************************************************************************/
#include "sbiib.h"

/****************************************************************************

 Function  : SBIIB_Init

 Purpose   : This function initialises the SBIIB card handle.

 Input(s)  : card        - handle for the card
             cardAddress - Address of VME card (Not used for PCI)

 Output(s) : error       - Function status

****************************************************************************/
Error  SBIIB_Init (CardHandle  *card,             /* Card handle           */
                   Ulong       cardAddress)       /* VME card address      */
/***************************************************************************/
{
Error  error;
#if(SBIIB_PCI_PLATFORM)
DWORD  dCntr;
#endif

    error = E_NO_ERROR;

    if(card == NO_POINTER)
        error = E_INVALID_POINTER;
    else
    {
        /********************************************************************
        *            If SBIIB PCI card then initialise the drivers          *
        ********************************************************************/
       #if(SBIIB_PCI_PLATFORM)
            WD_CardUnregister(card->hDRVR, &(card->cardReg));
            WD_Close(card->hDRVR);
            card->hDRVR = WD_Open();

            if(card->hDRVR == INVALID_HANDLE_VALUE)
                error = E_DRIVER_INIT_FAILED;

            if(error == E_NO_ERROR)
            {
                WD_Version(card->hDRVR, &(card->verBuf));
                if(card->verBuf.dwVer < WD_VER)
                    error = E_DRIVER_INIT_FAILED;
            };

            if(error == E_NO_ERROR)
            {
                strcpy(card->lic.cLicense,
                       "75d7dbfcfaee16c112eeb61a7a.Western Avionics Ltd.");
                WD_License(card->hDRVR, &(card->lic));
            }

            if(error == E_NO_ERROR)
            {
                card->pciScan.searchId.dwVendorId = MY_VENDOR_ID;
                card->pciScan.searchId.dwDeviceId = MY_DEVICE_ID;
                WD_PciScanCards(card->hDRVR, &(card->pciScan));
                if(card->pciScan.dwCards == 0)
                    error = E_DRIVER_INIT_FAILED;
            }

            if(error == E_NO_ERROR)
            {
                card->pciCardInfo.pciSlot = card->pciScan.cardSlot[0];
                WD_PciGetCardInfo(card->hDRVR, &(card->pciCardInfo));
                card->cardReg.Card = card->pciCardInfo.Card;

                card->G_resMemory    = INVALID_RES;
                card->G_resInterrupt = INVALID_RES;

                for(dCntr=0; dCntr < card->cardReg.Card.dwItems; dCntr++)
                {
                    switch(card->cardReg.Card.Item[dCntr].item)
                    {
                        case ITEM_INTERRUPT:
                            if(card->G_resInterrupt == INVALID_RES)
                                card->G_resInterrupt = dCntr;
                            else
                                error = E_DRIVER_INIT_FAILED;
                            break;

                        case ITEM_MEMORY:
                            if(card->G_resMemory == INVALID_RES)
                                card->G_resMemory = dCntr;
                            else
                                error = E_DRIVER_INIT_FAILED;
                            break;

                        case ITEM_IO:
                            error = E_DRIVER_INIT_FAILED;
                            break;
                    }
                }
            }

            if(error == E_NO_ERROR)
            {
                if(card->G_resMemory == INVALID_RES)
                    error = E_DRIVER_INIT_FAILED;
            }

            if(error == E_NO_ERROR)
            {
                card->cardReg.fCheckLockOnly = FALSE;
                WD_CardRegister(card->hDRVR, &(card->cardReg));

                if(card->cardReg.hCard == 0)
                    error = E_DRIVER_INIT_FAILED;
            }

            card->pciAddress =
               card->cardReg.Card.Item[card->G_resMemory].I.Mem.dwTransAddr;
        #endif

        /********************************************************************
        *        If SBIIB VME card then initialise the VME pointer          *
        ********************************************************************/
        card->vmeAddress = (Uword *)cardAddress;
    }

return (error);
}

/****************************************************************************

 Function  : SBIIB_RdClock

 Purpose   : This function reads the SBIIB clock value.

 Input(s)  : card   - handle for the card
             time   - TimeTag structure for result

 Output(s) : error  - Function status

****************************************************************************/
Error  SBIIB_RdClock(CardHandle  *card,           /* Card handle           */
                     TimeTag     *time)           /* TimeTag structure     */
/***************************************************************************/
{
Error  error;
Ulong  clkH, clkL;
Uword  ckH, ckM, ckL, ckT;

    error = E_NO_ERROR;

    /*
     If PCI card, init the read of clock by accessing the control register,
     wait 0.5us+ then read the clock as two 32 bit reads.
    */
    #if(SBIIB_PCI_PLATFORM)
        wW(card, CONTROL_REG, PCI_CLOCK_READ);
        for(clkH=0; clkH != 100; clkH++);
        card->trans.cmdTrans = RM_DWORD;
        card->trans.dwPort = (DWORD)CLOCK_ADDR + (DWORD)card->pciAddress;
        WD_Transfer(card->hDRVR, &(card->trans));
        clkH = (Ulong)card->trans.Data.Dword;
        card->trans.cmdTrans = RM_DWORD;
        card->trans.dwPort = (DWORD)CLOCK_ADDR + (DWORD)card->pciAddress;
        WD_Transfer(card->hDRVR, &(card->trans));
        clkL = (Ulong)card->trans.Data.Dword;
        ckH  = (Uword)((clkH >> 16) & 0xFFFF);
        ckM  = (Uword)(clkH & 0xFFFF);
        ckL  = (Uword)((clkL >> 16) & 0xFFFF);
        ckT  = (Uword)(clkL & 0xFFFF);
   #endif

    /*
     If VME card, init the read of clock by reading clock base address,
     wait 0.5us+ then read the clock as four 16 bit reads.
    */
    #if(SBIIB_VME_PLATFORM)
        ckH = rW(card, CLOCK_ADDR);
        for(clkH=0; clkH != 100; clkH++);
        ckH = rW(card, CLOCK_ADDR+2)
        ckM = rW(card, CLOCK_ADDR+2)
        ckL = rW(card, CLOCK_ADDR+2)
        ckT = rW(card, CLOCK_ADDR+2)
    #endif

    /*
     Now format the time value and save in the TimeTag structure.
    */
    if(error == E_NO_ERROR)
        error = SBIIB_FormatTtag(ckH, ckM, ckL, ckT, time);

return (error);
}

/****************************************************************************

 Function  : wW

 Purpose   : SBIIB primitive write word to card

 Input(s)  : card   - Handle for the card
             offset - Offset address in card
             value  - Value to write

 Output(s) : void

****************************************************************************/
void   wW(CardHandle  *card,                     /* Card handle            */
          Ulong       offset,                    /* Card address offset    */
          Uword       value)                     /* Value to write         */
/***************************************************************************/
{
Ulong  adoffs;

/****************************************************************************
*                         Here for SBIIB PCI card                           *
****************************************************************************/
    #if(SBIIB_PCI_PLATFORM)
        adoffs = offset;
        adoffs ^= 2;
        card->trans.cmdTrans = WM_WORD;
        card->trans.dwPort = (DWORD)adoffs + (DWORD)card->pciAddress;
        card->trans.Data.Word = value;
        WD_Transfer(card->hDRVR, &(card->trans));
    #endif

/****************************************************************************
*                         Here for SBIIB VME card                           *
****************************************************************************/
    #if(SBIIB_VME_PLATFORM)
        card->vmeAddress[offset/2] = value;
    #endif
}

/****************************************************************************

 Function  : rW

 Purpose   : SBIIB primitive read word from card

 Input(s)  : card   - Handle for the card
             offset - Offset address in card

 Output(s) : Uword  - Value at address in card.

****************************************************************************/
Uword  rW(CardHandle  *card,                     /* Card handle            */
          Ulong       offset)                    /* Card address offset    */
/***************************************************************************/
{
Ulong  adoffs;
Uword  value;

/****************************************************************************
*                         Here for SBIIB PCI card                           *
****************************************************************************/
    #if(SBIIB_PCI_PLATFORM)
        adoffs = offset;
        adoffs ^= 2;
        card->trans.cmdTrans = RM_WORD;
        card->trans.dwPort = (DWORD)adoffs + (DWORD)card->pciAddress;
        WD_Transfer(card->hDRVR, &(card->trans));
        value = (Uword)card->trans.Data.Word;
    #endif

/****************************************************************************
*                         Here for SBIIB VME card                           *
****************************************************************************/
    #if(SBIIB_VME_PLATFORM)
        value = card->vmeAddress[offset/2];
    #endif

return (value);
}
