/****************************************************************************

   Copyright (c) 2000, Western Avionics LTD.             All Rights Reserved.

*****************************************************************************

   File          : sb_ldbbc.c

   Author        : S.J.Buckley

   Date          : 11-01-2000

   Function      : SBIIB LDB_BCT functions.

   Update        : dd-mm-yy
   By            : S.J.Buckley
   Reason        :

****************************************************************************/
#include "sbiib.h"

/****************************************************************************

 Function  : SBIIB_WrProg

 Purpose   : This function writes the LDB_BCT background program

 Input(s)  : card       - handle for the card
             info       - Program information structure

 Output(s) : error      - Function status

****************************************************************************/
Error  SBIIB_WrProg(CardHandle        *card,      /* Card handle           */
                    LDB_BCT_PROGinfo  *info)      /* Program Info          */
/***************************************************************************/
{
Error  error;
Uword  i;
Ulong  addr, sVal;

    error = E_NO_ERROR;

    /*
     Test if parameters and operating modes are correct.
    */
    if(card == NO_POINTER || info == NO_POINTER)
        error = E_INVALID_POINTER;
    else
    if(card->mode != LDB_BCT_MODE)
        error = E_INVALID_MODE;
    else
    if(error == E_NO_ERROR && info->mCount > MSG_COUNT)
        error = E_INVALID_PARAMETER;
    else
    {
        for(i=0; i!=info->mCount; i++)
        {
            if(info->msgs[i] == 0 || info->msgs[i] > MSG_COUNT)
                error = E_INVALID_PARAMETER;
        }
    }

    /*
     Now write out the new frame.
    */
    if(error == E_NO_ERROR)
    {
        /*
         Set the loop count and frame duration
        */
        addr = info->fTime;
        wW(card, LDB_PROGRAM+2, info->fCount);
        wW(card, LDB_PROGRAM+6, (Uword)((addr >> 16) & 0xFFFF));
        wW(card, LDB_PROGRAM+8, (Uword)(addr & 0xFFFF));
        addr = LDB_PROGRAM+22;

        /*
         Set the MDB address for each SMB based on the message number.
        */
        for(i=0; i!=info->mCount; i++)
        {
            wW(card, addr, LDB_SMB);
            addr += 2;
            sVal = (Ulong)(info->msgs[i] - 1) * MDB_SZ;
            sVal = (sVal + MDB_START) & 0xFFFF;
            wW(card, addr, (Uword)sVal);
            wW(card, addr+2, (Uword)(addr+4));
            addr += 4;
        }

        /*
         Now set the BRA or DBNE and the branched offset.
        */
        if(info->fCount == LDB_TX_FOREVER)
            wW(card, addr, LDB_BRA);
        else
            wW(card, addr, LDB_DBNE);

        sVal = 0x10000 + LDB_PROGRAM + 10 - addr;
        wW(card, addr+2, (Uword)(sVal & 0xFFFF));
        wW(card, addr+4, LDB_HALT);
    }

return (error);
}

/****************************************************************************

 Function  : SBIIB_RdPrg

 Purpose   : This function reads back the LDB_BCT background program

 Input(s)  : card       - handle for the card
             info       - Program information structure

 Output(s) : error      - Function status

****************************************************************************/
Error  SBIIB_RdProg(CardHandle        *card,      /* Card handle           */
                    LDB_BCT_PROGinfo  *info)      /* Program Info          */
/***************************************************************************/
{
Error  error;
Ulong  addr, i, mNo;

    error = E_NO_ERROR;

    /*
     Test if parameters and operating modes are correct.
    */
    if(card == NO_POINTER || info == NO_POINTER)
        error = E_INVALID_POINTER;
    else
    if(card->mode != LDB_BCT_MODE)
        error = E_INVALID_MODE;

    /*
     Now read the frame information.
    */
    else
    {
        /*
         Read the loop count and frame duration.
        */
        info->fCount = rW(card, LDB_PROGRAM+2);
        addr = (Ulong)rW(card, LDB_PROGRAM+6);
        info->fTime  = (addr << 16) & 0xFFFF;
        info->fTime |= (Ulong)rW(card, LDB_PROGRAM+8) & 0xFFFF;
        addr = LDB_PROGRAM+22;

        /*
         Set the MSG numbers in the array based on the MDB address.
        */
        for(i=0, info->mCount=0; rW(card, addr)==LDB_SMB; addr+=6, i++)
        {
            mNo  = (Ulong)rW(card, addr+2);
            mNo -= MDB_START;
            mNo  = (mNo/MDB_SZ) + 1;
            info->msgs[i] = (Uword)mNo;
            (info->mCount)++;
        }
    }

return (error);
}

/****************************************************************************

 Function  : SBIIB_WrMsg

 Purpose   : This function writes a LDB_BCT message

 Input(s)  : card       - handle for the card
             info       - Message information structure

 Output(s) : error      - Function status

****************************************************************************/
Error  SBIIB_WrMsg(CardHandle        *card,       /* Card handle           */
                   LDB_BCT_MSGinfo   *info)       /* Message Info          */
/***************************************************************************/
{
Error  error;
Uword  i, wordNo;
Ulong  addr;

    /*
     Test if parameters and operating modes are correct.
    */
    if(card == NO_POINTER || info == NO_POINTER)
        error = E_INVALID_POINTER;
    else
    if(card->mode != LDB_BCT_MODE)
        error = E_INVALID_MODE;
    else
    if(info->msgNo == 0 || info->msgNo > MSG_COUNT)
        error = E_INVALID_PARAMETER;
    else
    if(info->msgType != RX_MESSAGE && info->msgType != TX_MESSAGE)
        error = E_INVALID_PARAMETER;
    else
    if(info->dataCount > MAX_DATA_SZ)
        error = E_INVALID_PARAMETER;
    else
    if(info->sevTest != SEV_TEST && info->sevTest != SEV_NO_TEST)
        error = E_INVALID_PARAMETER;
    else
    if(info->busID != PRIMARY_BUS && info->busID != SECONDARY_BUS)
        error = E_INVALID_PARAMETER;
    else
    if(info->data != NO_POINTER && info->data->buffSZ > MAX_DATA_SZ)
        error = E_INVALID_PARAMETER;
    else
    if(info->data != NO_POINTER && info->data->wCount > MAX_DATA_SZ)
        error = E_INVALID_PARAMETER;
    else
    {
        addr  = (Ulong)((info->msgNo - 1) * MDB_SZ) + MDB_START;
        error = SBIIB_SetErr(card, addr+LDB_MDB_TXERRS, info->txErrs);
    }

    /*
     Fill in the MDB with the input variables.
    */
    if(error == E_NO_ERROR)
    {
        wW(card, addr+LDB_MDB_IRQCON, info->irqDef);
        i = (Uword)((info->command >> 16) & 0x00FF) | info->busID;
        wW(card, addr+LDB_MDB_CMNDHI, i);
        i = (Uword)(info->command & 0xFFFF);
        wW(card, addr+LDB_MDB_CMNDLO, i);
        i = info->dataCount;
        if(info->sevTest == SEV_TEST)
            i |= 0x4000;
        if(info->msgType == TX_MESSAGE)
            i |= 0x8000;
        wW(card, addr+LDB_MDB_DATAWC, i);
        i = (Uword)((info->gapTime >> 16) & 0xFFFF);
        wW(card, addr+LDB_MDB_GAPTHI, i);
        i = (Uword)(info->gapTime & 0xFFFF);
        wW(card, addr+LDB_MDB_GAPTLO, i);
    }

    /*
     If no error and data required, fill in the data buffer variables
     and then copy the data array into the buffer.
    */
    if(error == E_NO_ERROR && info->data != NO_POINTER)
    {
        addr = (Ulong)(info->msgNo - 1);
        addr = (Ulong)(addr * DATA_BLOCK_SZ) + DATA_BLOCKS;
        wW(card, addr+LDB_DATA_TTAGHI, 0);
        wW(card, addr+LDB_DATA_TTAGMD, 0);
        wW(card, addr+LDB_DATA_TTAGLO, 0);
        wW(card, addr+LDB_DATA_TTAGTK, 0);
        wW(card, addr+LDB_DATA_BUFFSZ, info->data->buffSZ);
        wW(card, addr+LDB_DATA_STATUS, 0);
        wW(card, addr+LDB_DATA_WORDCT, 0);
        wW(card, addr+LDB_DATA_CHKSUM, 0);

        addr += LDB_DATA_FIRSTD;

        for(wordNo=0; wordNo!= info->data->wCount; wordNo++)
        {
            i = (Uword)((info->data->data[wordNo] >> 16) & 0xFFFF);
            wW(card, addr, i);
            i = (Uword)(info->data->data[wordNo] & 0xFFFF);
            wW(card, addr+2, i);
            addr += 4;
        }
    }

return (error);
}

/****************************************************************************

 Function  : SBIIB_RdMsg

 Purpose   : This function reads a LDB_BCT message

 Input(s)  : card       - handle for the card
             info       - Message information structure

 Output(s) : error      - Function status

****************************************************************************/
Error  SBIIB_RdMsg(CardHandle        *card,       /* Card handle           */
                   LDB_BCT_MSGinfo   *info)       /* Message Info          */
/***************************************************************************/
{
Error  error;
Uword  i, wordNo, wCnt;
Ulong  addr, hVal, lVal;

    /*
     Test if parameters and operating modes are correct.
    */
    if(card == NO_POINTER || info == NO_POINTER)
        error = E_INVALID_POINTER;
    else
    if(card->mode != LDB_BCT_MODE)
        error = E_INVALID_MODE;
    else
    if(info->msgNo == 0 || info->msgNo > MSG_COUNT)
        error = E_INVALID_PARAMETER;
    else
    if(info->data != NO_POINTER && info->data->wCount > MAX_DATA_SZ)
        error = E_INVALID_PARAMETER;
    else
    {
        addr   = (Ulong)((info->msgNo - 1) * MDB_SZ) + MDB_START;
        error  = SBIIB_GetErr(card, addr+LDB_MDB_TXERRS, info->txErrs);
    }

    /*
     Get the MDB variables into the structure.
    */
    if(error == E_NO_ERROR)
    {
        info->irqDef = rW(card, addr+LDB_MDB_IRQCON);
        hVal = (Ulong)rW(card, addr+LDB_MDB_CMNDHI) & 0xFFFF;
        if((hVal & 0x8000) != 0)
            info->busID = PRIMARY_BUS;
        else
            info->busID = SECONDARY_BUS;
        info->command = (hVal << 16) & 0x00FF0000;
        hVal = (Ulong)rW(card, addr+LDB_MDB_CMNDLO) & 0xFFFF;
        info->command |= hVal;

        i = rW(card, addr+LDB_MDB_DATAWC);

        if((i & 0x8000) != 0)
            info->msgType = TX_MESSAGE;
        else        
            info->msgType = RX_MESSAGE;
        if((i & 0x4000) != 0)
            info->sevTest = SEV_TEST;
        else        
            info->sevTest = SEV_NO_TEST;

        i &= 0x03FF;
        info->dataCount = i;
        
        hVal = (Ulong)rW(card, addr+LDB_MDB_GAPTHI) & 0xFFFF;
        info->gapTime = hVal << 16;
        hVal = (Ulong)rW(card, addr+LDB_MDB_GAPTLO) & 0xFFFF;
        info->gapTime |= hVal;
    }

    /*
     If no error and data required, read the data buffer variables
     and then copy the data buffer into the array.
    */
    if(error == E_NO_ERROR && info->data != NO_POINTER)
    {
        wCnt = info->data->wCount;
        addr = (Ulong)(info->msgNo - 1);
        addr = (Ulong)(addr * DATA_BLOCK_SZ) + DATA_BLOCKS;

        /*
         Get timetag if required.
        */
        if(&info->data->time != NO_POINTER)
            error = SBIIB_RdTtag(card, addr+LDB_DATA_TTAGHI, &info->data->time);
            
        info->data->buffSZ = rW(card, addr+LDB_DATA_BUFFSZ);
        info->data->status = rW(card, addr+LDB_DATA_STATUS);
        info->data->wCount = rW(card, addr+LDB_DATA_WORDCT);
        info->data->chksum = rW(card, addr+LDB_DATA_CHKSUM);

        addr += LDB_DATA_FIRSTD;

        for(wordNo=0; wordNo!=wCnt; wordNo++)
        {
            hVal = (Ulong)rW(card, addr) & 0xFFFF;
            hVal <<= 16;
            lVal = (Ulong)rW(card, addr+2) & 0xFFFF;
            hVal |= lVal;
            info->data->data[wordNo] = hVal;           
            addr += 4;
        }
    }

return (error);
}

/****************************************************************************

 Function  : SBIIB_RdLdbBcQ

 Purpose   : This function reads an entry of the LDB_BCT Queue.

 Input(s)  : card    - handle for the card
             qData   - LDB_BCT_Qdata structure for storing result

 Output(s) : error   - Function status

****************************************************************************/
Error  SBIIB_RdLdbBcQ(CardHandle     *card,       /* Card handle           */
                      LDB_BCT_Qdata  *qData)      /* BC Queue structure    */
/***************************************************************************/
{
Error  error;
Ulong  aVal, tVal, offset;

    error  = E_NO_ERROR;
    offset = qData->offset * LDB_BCT_Q_ELEMENT_SZ;

    /*
     Test if parameters values are good.
    */
    if(card == NO_POINTER || qData == NO_POINTER)
        error = E_INVALID_POINTER;
    else
    if(card->mode != LDB_BCT_MODE)
        error = E_INVALID_MODE;
    else
    if(qData->priority != HI_PRIORITY_Q && qData->priority != LO_PRIORITY_Q)
        error = E_INVALID_PARAMETER;
    else
    if(offset >= LDB_Q_SZ)
        error = E_INVALID_PARAMETER;

    /*
     If clear the queue, reset offset and clear all queue data.
    */
    else
    if(qData->action == CLEAR_Q)
    {
        if(qData->priority == HI_PRIORITY_Q)
            aVal = HI_PRI_Q;
        else
            aVal = LO_PRI_Q;

        qData->offset = 0;
        for(tVal=0; tVal != LDB_Q_SZ; tVal+=2)
            wW(card, aVal+tVal, 0);
        qData->result = NO_Q_ELEMENT;
    }

    /*
     Test for  element in queue.  If true, clear it, read in the time tag and
     read  in the  data  HI and LO values.  Finally increment  the  Q pointer
     accounting for wrap-round.
    */
    else
    if(qData->action == READ_Q || qData->action == READ_Q_AND_CLEAR)
    {
        if(qData->priority == HI_PRIORITY_Q)
            aVal = HI_PRI_Q + offset;
        else
            aVal = LO_PRI_Q + offset;

        qData->messNo =	rW(card, aVal+2);

        if(qData->messNo == 0)
            qData->result = NO_Q_ELEMENT;
        else
        {
            if(qData->action == READ_Q_AND_CLEAR)
                wW(card, aVal+2, 0);
            qData->result = GOT_Q_ELEMENT;
            qData->errors = rW(card, aVal);
            offset += LDB_BCT_Q_ELEMENT_SZ;
            offset &= (LDB_Q_SZ - LDB_BCT_Q_ELEMENT_SZ);
            qData->offset = offset/LDB_BCT_Q_ELEMENT_SZ;
        }
    }
    else
        error = E_INVALID_PARAMETER;

return (error);
}

/****************************************************************************

 Function  : SBIIB_RunLdbBc

 Purpose   : This function runs the LDB_BCT.

 Input(s)  : card    - handle for the card
             count   - Frame count to TX

 Output(s) : error   - Function status

****************************************************************************/
Error  SBIIB_RunLdbBc(CardHandle    *card,        /* Card handle           */
                      Uword         count)        /* TX count              */
/***************************************************************************/
{
Error  error;
Ulong  addr;

    /*
     Test if parameters and operating modes are correct.
    */
    if(card == NO_POINTER)
        error = E_INVALID_POINTER;
    else
    if(card->mode != LDB_BCT_MODE)
        error = E_INVALID_MODE;

    /*
     Now set the count and get to the end of the frame. If TX_FOREVER set the
     loop on a  LDB_BRA  else set for a LDB_DBNE.  Finally start  the LDB_BCT
     transmitting.
    */
    else
    {
        wW(card, LDB_PROGRAM+2, count);
        addr = LDB_PROGRAM+22;

        while(rW(card, addr) == LDB_SMB)
            addr+=6;

        if(count == LDB_TX_FOREVER)
            wW(card, addr, LDB_BRA);
        else
            wW(card, addr, LDB_DBNE);

        wW(card, addr+4, LDB_HALT);

        error = SBIIB_Command(card, START_CARD);
    }

return (error);
}
