/****************************************************************************

   Copyright (c) 2000, Western Avionics LTD.             All Rights Reserved.

*****************************************************************************

   File          : sb_ldbrt.c

   Author        : S.J.Buckley

   Date          : 11-01-2000

   Function      : SBIIB LDB_MRT functions.

   Update        : dd-mm-yy
   By            : S.J.Buckley
   Reason        :

****************************************************************************/
#include "sbiib.h"

/****************************************************************************

 Function  : SBIIB_WrLdbRT

 Purpose   : This function writes the LDB_RT parameters.

 Input(s)  : card       - handle for the card
             info       - Program information structure

 Output(s) : error      - Function status

****************************************************************************/
Error  SBIIB_WrLdbRT(CardHandle       *card,      /* Card handle           */
                     LDB_MRTinfo      *info)      /* LDB_MRT Info          */
/***************************************************************************/
{
Error  error;
Ulong  addr, lVal, i;
Uword  wVal, wordNo;

    error = E_NO_ERROR;

    /*
     Test if parameters and operating modes are correct.
    */
    if(card == NO_POINTER || info == NO_POINTER)
        error = E_INVALID_POINTER;
    else
    if(card->mode != LDB_MRT_MODE)
        error = E_INVALID_MODE;
    else
    if(info->rtNo >= (Uword)RT_COUNT)
        error = E_INVALID_PARAMETER;
    else
    if(info->bus != LDB_MRT_SELECT_PRI &&
       info->bus != LDB_MRT_SELECT_SEC &&
       info->bus != LDB_MRT_SELECT_BOTH)
        error = E_INVALID_PARAMETER;
    else
    if(info->mode != LDB_MRT_ACTIVE  &&
       info->mode != LDB_MRT_PASSIVE)
        error = E_INVALID_PARAMETER;
    else
    if(info->action != NO_DATA_AND_READY      &&
       info->action != NO_DATA_NOT_READY      &&
       info->action != CRITICAL_DATA          &&
       info->action != NON_CRITICAL_AND_READY &&
       info->action != NON_CRITICAL_NOT_READY)
        error = E_INVALID_PARAMETER;
    else
    if(info->updateState != UPDATE_STATE_ON   &&
       info->updateState != UPDATE_STATE_OFF)
        error = E_INVALID_PARAMETER;
    else
    if(info->ddtCount > LDB_MRT_DDT_MASK)
        error = E_INVALID_PARAMETER;
    else
    if(info->state != WFI   && info->state != WFGAH &&
       info->state != WFHCD && info->state != WFS   &&
       info->state != WFSR  && info->state != WFSRE)
        error = E_INVALID_PARAMETER;
    else
    if(info->wrap != WRAP && info->wrap != NO_WRAP)
        error = E_INVALID_PARAMETER;
    else
    if(info->txData != NO_POINTER && info->txData->buffSZ > MAX_DATA_SZ)
        error = E_INVALID_PARAMETER;
    else
    if(info->txData != NO_POINTER && info->txData->wCount > MAX_DATA_SZ)
        error = E_INVALID_PARAMETER;
    else
    if(info->rxData != NO_POINTER && info->rxData->buffSZ > MAX_DATA_SZ)
        error = E_INVALID_PARAMETER;
    else
    if(info->rxData != NO_POINTER && info->rxData->wCount > MAX_DATA_SZ)
        error = E_INVALID_PARAMETER;
    else
        addr = RT_SIM_TABLE;

    /*
     Now test and set the TX error information.
    */
    if(error == E_NO_ERROR)
        error = SBIIB_SetErr(card, addr+LDB_SIM_GAHERR, info->GAHerrs);
    if(error == E_NO_ERROR)
        error = SBIIB_SetErr(card, addr+LDB_SIM_INDERR, info->INDerrs);
    if(error == E_NO_ERROR)
        error = SBIIB_SetErr(card, addr+LDB_SIM_IWDERR, info->IWDerrs);
    if(error == E_NO_ERROR)
        error = SBIIB_SetErr(card, addr+LDB_SIM_SRQERR, info->SRQerrs);
    if(error == E_NO_ERROR)
        error = SBIIB_SetErr(card, addr+LDB_SIM_FRCERR, info->FRCerrs);

    /*
     Now set the SIM table variables.
    */
    if(error == E_NO_ERROR)
    {
        wVal = info->mode|info->action|info->ddtCount;
        wW(card, addr+LDB_SIM_MODEWD, wVal);
        wW(card, addr+LDB_SIM_IRQWDA, info->irqDefA);
        wW(card, addr+LDB_SIM_IRQWDB, info->irqDefB);
        wW(card, addr+LDB_SIM_TTAGHI, 0);
        wW(card, addr+LDB_SIM_TTAGMD, 0);
        wW(card, addr+LDB_SIM_TTAGLO, 0);
        wW(card, addr+LDB_SIM_TTAGTK, 0);
        wW(card, addr+LDB_SIM_LCMDHI, 0);
        wW(card, addr+LDB_SIM_LCMDLO, 0);
        wW(card, addr+LDB_SIM_LRESHI, 0);
        wW(card, addr+LDB_SIM_LRESLO, 0);

        if(info->updateState == UPDATE_STATE_ON)
            wW(card, addr+LDB_SIM_STATEM, info->state);

        /*
         Now set the TXRX buffers for wrap round or unique.
        */
        if(info->wrap == WRAP)
        {
            wVal = rW(card, addr+LDB_SIM_TXBFHI);
            wW(card, addr+LDB_SIM_RXBFHI, wVal);
            wVal = rW(card, addr+LDB_SIM_TXBFLO);
            wW(card, addr+LDB_SIM_RXBFLO, wVal);
        }
        else
        {
            lVal  = (Ulong)rW(card, addr+LDB_SIM_TXBFHI);
            lVal  = (lVal << 16) & 0xFFFF0000;
            lVal |= (Ulong)rW(card, addr+LDB_SIM_TXBFLO) & 0xFFFF;
            lVal += DATA_BLOCK_SZ;
            wW(card, addr+LDB_SIM_RXBFHI, (Uword)((lVal >> 16) & 0xFFFF));
            wW(card, addr+LDB_SIM_RXBFLO, (Uword)(lVal & 0xFFFF));
        }
    }

    /*
     If no error and TX data required, fill in the data buffer variables
     and then copy the data array into the buffer.
    */
    if(error == E_NO_ERROR && info->txData != NO_POINTER)
    {

        lVal  = (Ulong)rW(card, addr+LDB_SIM_TXBFHI);
        lVal  = (lVal << 16) & 0xFFFF0000;
        lVal |= (Ulong)rW(card, addr+LDB_SIM_TXBFLO) & 0xFFFF;

        wW(card, lVal+LDB_DATA_TTAGHI, 0);
        wW(card, lVal+LDB_DATA_TTAGMD, 0);
        wW(card, lVal+LDB_DATA_TTAGLO, 0);
        wW(card, lVal+LDB_DATA_TTAGTK, 0);
        wW(card, lVal+LDB_DATA_BUFFSZ, info->txData->buffSZ);
        wW(card, lVal+LDB_DATA_STATUS, 0);
        wW(card, lVal+LDB_DATA_WORDCT, 0);
        wW(card, lVal+LDB_DATA_CHKSUM, 0);

        lVal += LDB_DATA_FIRSTD;

        for(wordNo=0; wordNo != info->txData->wCount; lVal+=4, wordNo++)
        {
            i = (info->txData->data[wordNo] >> 16) & 0xFFFF;
            wW(card, lVal, (Uword)i);
            i = info->txData->data[wordNo] & 0xFFFF;
            wW(card, lVal+2, (Uword)i);
        }
    }

    /*
     If no error and RX data required, fill in the data buffer variables
     and then copy the data array into the buffer.
    */
    if(error == E_NO_ERROR && info->rxData != NO_POINTER)
    {
        lVal  = (Ulong)rW(card, addr+LDB_SIM_RXBFHI);
        lVal  = (lVal << 16) & 0xFFFF0000;
        lVal |= (Ulong)rW(card, addr+LDB_SIM_RXBFLO) & 0xFFFF;

        wW(card, lVal+LDB_DATA_TTAGHI, 0);
        wW(card, lVal+LDB_DATA_TTAGMD, 0);
        wW(card, lVal+LDB_DATA_TTAGLO, 0);
        wW(card, lVal+LDB_DATA_TTAGTK, 0);
        wW(card, lVal+LDB_DATA_BUFFSZ, info->rxData->buffSZ);
        wW(card, lVal+LDB_DATA_STATUS, 0);
        wW(card, lVal+LDB_DATA_WORDCT, 0);
        wW(card, lVal+LDB_DATA_CHKSUM, 0);

        lVal += LDB_DATA_FIRSTD;

        for(wordNo=0; wordNo != info->rxData->wCount; lVal+=4, wordNo++)
        {
            i = (info->rxData->data[wordNo] >> 16) & 0xFFFF;
            wW(card, lVal, (Uword)i);
            i = info->rxData->data[wordNo] & 0xFFFF;
            wW(card, lVal+2, (Uword)i);
        }
    }

    /*
     Now set the RT number in the base register.
    */
    if(error == E_NO_ERROR)
        wW(card, LDB_RT_ADDRESS_REG, info->rtNo);

    /*
     Now set the RT bus select base register.
    */
    if(error == E_NO_ERROR)
    {
        wVal = rW(card, BUS_SELECT_REG);
        wVal = (wVal & 0xFFFC) | info->bus;
        wW(card, BUS_SELECT_REG, wVal);
    }

return (error);
}

/****************************************************************************

 Function  : SBIIB_RdLdbRT

 Purpose   : This function reads the LDB_RT parameters.

 Input(s)  : card       - handle for the card
             info       - Program information structure

 Output(s) : error      - Function status

****************************************************************************/
Error  SBIIB_RdLdbRT(CardHandle       *card,      /* Card handle           */
                     LDB_MRTinfo      *info)      /* LDB_MRT Info          */
/***************************************************************************/
{
Error  error;
Ulong  addr, lVal1, lVal2, lVal3;
Uword  wVal, wCnt, wordNo;

    error = E_NO_ERROR;

    /*
     Test if parameters and operating modes are correct.
    */
    if(card == NO_POINTER || info == NO_POINTER)
        error = E_INVALID_POINTER;
    else
    if(card->mode != LDB_MRT_MODE)
        error = E_INVALID_MODE;
    else
    if(info->txData != NO_POINTER && info->txData->wCount > MAX_DATA_SZ)
        error = E_INVALID_PARAMETER;
    else
    if(info->rxData != NO_POINTER && info->rxData->wCount > MAX_DATA_SZ)
        error = E_INVALID_PARAMETER;

    addr = RT_SIM_TABLE;

    /*
     Now test and set the TX error and SIM TTAG information.
    */
    if(error == E_NO_ERROR)
        error = SBIIB_GetErr(card, addr+LDB_SIM_GAHERR, info->GAHerrs);
    if(error == E_NO_ERROR)
        error = SBIIB_GetErr(card, addr+LDB_SIM_INDERR, info->INDerrs);
    if(error == E_NO_ERROR)
        error = SBIIB_GetErr(card, addr+LDB_SIM_IWDERR, info->IWDerrs);
    if(error == E_NO_ERROR)
        error = SBIIB_GetErr(card, addr+LDB_SIM_SRQERR, info->SRQerrs);
    if(error == E_NO_ERROR)
        error = SBIIB_GetErr(card, addr+LDB_SIM_FRCERR, info->FRCerrs);
    if(error == E_NO_ERROR && &info->time != NO_POINTER)
        error = SBIIB_RdTtag(card, addr+LDB_SIM_TTAGHI, &info->time);

    /*
     Now set the SIM table variables.
    */
    if(error == E_NO_ERROR)
    {
        wVal = rW(card, addr+LDB_SIM_MODEWD);
        info->mode     = wVal & LDB_MRT_MODE_MASK;
        info->action   = wVal & LDB_MRT_ACTION_MASK;
        info->ddtCount = wVal & LDB_MRT_DDT_MASK;
        info->state    = rW(card, addr+LDB_SIM_STATEM);
        info->irqDefA  = rW(card, addr+LDB_SIM_IRQWDA);
        info->irqDefB  = rW(card, addr+LDB_SIM_IRQWDB);
        lVal1  = (Ulong)rW(card, addr+LDB_SIM_LCMDHI) & 0xFFFF;
        lVal1  = (lVal1 << 16) & 0xFFFF0000;
        lVal1 |= (Ulong)rW(card, addr+LDB_SIM_LCMDLO) & 0xFFFF;
        info->lastCmnd = lVal1;
        lVal1  = (Ulong)rW(card, addr+LDB_SIM_LRESHI) & 0xFFFF;
        lVal1  = (lVal1 << 16) & 0xFFFF0000;
        lVal1 |= (Ulong)rW(card, addr+LDB_SIM_LRESLO) & 0xFFFF;
        info->lastResp = lVal1;

        lVal1  = (Ulong)rW(card, addr+LDB_SIM_TXBFHI) & 0xFFFF;
        lVal1  = (lVal1 << 16) & 0xFFFF0000;
        lVal1 |= (Ulong)rW(card, addr+LDB_SIM_TXBFLO) & 0xFFFF;
        lVal2  = (Ulong)rW(card, addr+LDB_SIM_RXBFHI) & 0xFFFF;
        lVal2  = (lVal2 << 16) & 0xFFFF0000;
        lVal2 |= (Ulong)rW(card, addr+LDB_SIM_RXBFLO) & 0xFFFF;

        if(lVal1 == lVal2)
            info->wrap = WRAP;
        else
            info->wrap = NO_WRAP;
    }

    /*
     If no error and data required, read the TX data buffer variables
     and then copy the data buffer into the array.
    */
    if(error == E_NO_ERROR && info->txData != NO_POINTER)
    {
        wCnt = info->txData->wCount;

        lVal1  = (Ulong)rW(card, addr+LDB_SIM_TXBFHI);
        lVal1  = (lVal1 << 16) & 0xFFFF0000;
        lVal1 |= (Ulong)rW(card, addr+LDB_SIM_TXBFLO) & 0xFFFF;

        /*
         Get timetag if required.
        */
        if(&info->txData->time != NO_POINTER)
            error = SBIIB_RdTtag(card, lVal1+LDB_DATA_TTAGHI, &info->txData->time);

        info->txData->buffSZ = rW(card, lVal1+LDB_DATA_BUFFSZ);
        info->txData->status = rW(card, lVal1+LDB_DATA_STATUS);
        info->txData->wCount = rW(card, lVal1+LDB_DATA_WORDCT);
        info->txData->chksum = rW(card, lVal1+LDB_DATA_CHKSUM);

        lVal1 += LDB_DATA_FIRSTD;

        for(wordNo=0; wordNo!=wCnt; wordNo++)
        {
            lVal2 = (Ulong)rW(card, lVal1) & 0xFFFF;
            lVal2 <<= 16;
            lVal3 = (Ulong)rW(card, lVal1+2) & 0xFFFF;
            lVal2 |= lVal3;
            info->txData->data[wordNo] = lVal2;
            lVal1 += 4;
        }
    }

    /*
     If no error and data required, read the RX data buffer variables
     and then copy the data buffer into the array.
    */
    if(error == E_NO_ERROR && info->rxData != NO_POINTER)
    {
        wCnt = info->rxData->wCount;

        lVal1  = (Ulong)rW(card, addr+LDB_SIM_RXBFHI);
        lVal1  = (lVal1 << 16) & 0xFFFF0000;
        lVal1 |= (Ulong)rW(card, addr+LDB_SIM_RXBFLO) & 0xFFFF;

        /*
         Get timetag if required.
        */
        if(&info->rxData->time != NO_POINTER)
            error = SBIIB_RdTtag(card, lVal1+LDB_DATA_TTAGHI, &info->rxData->time);

        info->rxData->buffSZ = rW(card, lVal1+LDB_DATA_BUFFSZ);
        info->rxData->status = rW(card, lVal1+LDB_DATA_STATUS);
        info->rxData->wCount = rW(card, lVal1+LDB_DATA_WORDCT);
        info->rxData->chksum = rW(card, lVal1+LDB_DATA_CHKSUM);

        lVal1 += LDB_DATA_FIRSTD;

        for(wordNo=0; wordNo!=wCnt; wordNo++)
        {
            lVal2 = (Ulong)rW(card, lVal1) & 0xFFFF;
            lVal2 <<= 16;
            lVal3 = (Ulong)rW(card, lVal1+2) & 0xFFFF;
            lVal2 |= lVal3;
            info->rxData->data[wordNo] = lVal2;           
            lVal1 += 4;
        }
    }

    /*
     Now read the RT number in the base register.
    */
    if(error == E_NO_ERROR)
        info->rtNo = rW(card, LDB_RT_ADDRESS_REG);

    /*
     Now read the RT bus select base register.
    */
    if(error == E_NO_ERROR)
        info->bus = rW(card, BUS_SELECT_REG) & 3;

return (error);
}

/****************************************************************************

 Function  : SBIIB_RdLdbRTstate

 Purpose   : This function reads the state machine value of the RT.

 Input(s)  : card       - handle for the card
             state      - RT state machine value storage

 Output(s) : error      - Function status

****************************************************************************/
Error  SBIIB_RdLdbRTstate(CardHandle  *card,      /* Card handle           */
                          Uword       *state)     /* RT state machine      */
/***************************************************************************/
{
Error  error;
Ulong  addr;

    error = E_NO_ERROR;

    /*
     Test if parameters and operating modes are correct.
    */
    if(card == NO_POINTER || state == NO_POINTER)
        error = E_INVALID_POINTER;
    else
    if(card->mode != LDB_MRT_MODE)
        error = E_INVALID_MODE;

    /*
     Now read the state machine value.
    */
    else
        *state = rW(card, RT_SIM_TABLE+LDB_SIM_STATEM);

return (error);
}

/****************************************************************************

 Function  : SBIIB_WrLdbRTaction

 Purpose   : This function writes the LDB_RT action to the RT.

 Input(s)  : card       - handle for the card
             action     - RT action (CRITICAL etc.)

 Output(s) : error      - Function status

****************************************************************************/
Error  SBIIB_WrLdbRTaction(CardHandle  *card,     /* Card handle           */
                           Uword       action)    /* RT action value       */
/***************************************************************************/
{
Error  error;
Ulong  addr;
Uword  wVal;

    error = E_NO_ERROR;

    /*
     Test if parameters and operating modes are correct.
    */
    if(card == NO_POINTER)
        error = E_INVALID_POINTER;
    else
    if(card->mode != LDB_MRT_MODE)
        error = E_INVALID_MODE;
    else
    if(action != NO_DATA_AND_READY      &&
       action != NO_DATA_NOT_READY      &&
       action != CRITICAL_DATA          &&
       action != NON_CRITICAL_AND_READY &&
       action != NON_CRITICAL_NOT_READY)
        error = E_INVALID_PARAMETER;

    /*
     Now read the mode word, mask off old action and set the new one.
    */
    else
    {
        addr   = RT_SIM_TABLE;
        wVal   = rW(card, addr+LDB_SIM_MODEWD);
        wVal   = (wVal & LDB_MRT_ACTION_CLEAR) | action;
        wW(card, addr+LDB_SIM_MODEWD, wVal);
    }
 
return (error);
}

/****************************************************************************

 Function  : SBIIB_RdLdbRtQ

 Purpose   : This function reads an entry of the LDB_MRT Queue.

 Input(s)  : card    - handle for the card
             qData   - LDB_MRT_Qdata structure for storing result

 Output(s) : error   - Function status

****************************************************************************/
Error  SBIIB_RdLdbRtQ(CardHandle     *card,       /* Card handle           */
                      LDB_MRT_Qdata  *qData)      /* RT Queue structure    */
/***************************************************************************/
{
Error  error;
Ulong  aVal, tVal, offset;

    error  = E_NO_ERROR;
    offset = qData->offset * LDB_MRT_Q_ELEMENT_SZ;

    /*
     Test if parameters values are good.
    */
    if(card == NO_POINTER || qData == NO_POINTER)
        error = E_INVALID_POINTER;
    else
    if(card->mode != LDB_MRT_MODE)
        error = E_INVALID_MODE;
    else
    if(qData->priority != HI_PRIORITY_Q && qData->priority != LO_PRIORITY_Q)
        error = E_INVALID_PARAMETER;
    else
    if(offset > (LDB_Q_SZ - LDB_MRT_Q_ELEMENT_SZ))
        error = E_INVALID_PARAMETER;

    /*
     If clear the queue, reset offset and clear all queue data.
    */
    else
    if(qData->action == CLEAR_Q)
    {
        if(qData->priority == HI_PRIORITY_Q)
            aVal = HI_PRI_Q;
        else
            aVal = LO_PRI_Q;

        qData->offset = 0;
        for(tVal=0; tVal != LDB_Q_SZ; tVal+=2)
            wW(card, aVal+tVal, 0);
        qData->result = NO_Q_ELEMENT;
    }

    /*
     Test for  element in queue.  If true, clear it, read in the time tag and
     read  in the  data  HI and LO values.  Finally increment  the  Q pointer
     accounting for wrap-round.
    */
    else
    if(qData->action == READ_Q || qData->action == READ_Q_AND_CLEAR)
    {
        if(qData->priority == HI_PRIORITY_Q)
            aVal = HI_PRI_Q + offset;
        else
            aVal = LO_PRI_Q + offset;

        if(rW(card, aVal+6) == 0)
            qData->result = NO_Q_ELEMENT;
        else
        {
            qData->result  = GOT_Q_ELEMENT;
            qData->errors  = rW(card, aVal+4);
            qData->altData = rW(card, aVal+6);
            tVal  = (Ulong)rW(card, aVal);
            tVal  = (tVal << 16) & 0xFFFF0000;
            tVal |= (Ulong)rW(card, aVal+2) & 0xFFFF;
            qData->command = tVal;
            offset += LDB_MRT_Q_ELEMENT_SZ;
            offset &= (LDB_Q_SZ - LDB_MRT_Q_ELEMENT_SZ);
            qData->offset = offset/LDB_MRT_Q_ELEMENT_SZ;
            if(qData->action == READ_Q_AND_CLEAR)
                wW(card, aVal+6, 0);
        }
    }
    else
        error = E_INVALID_PARAMETER;

return (error);
}

/****************************************************************************

 Function  : SBIIB_RunLdbRt

 Purpose   : This function start the LDB_MRT running.

 Input(s)  : card    - handle for the card

 Output(s) : error   - Function status

****************************************************************************/
Error  SBIIB_RunLdbRt(CardHandle  *card,          /* Card handle           */
                      Uword       rtNo)           /* RT address number     */
/***************************************************************************/
{
Error  error;

    /*
     Test if parameters values are good anf if true run the LDB_BCT.
    */
    if(card == NO_POINTER)
        error = E_INVALID_POINTER;
    else
    if(card->mode != LDB_MRT_MODE)
        error = E_INVALID_MODE;
    else
    if(rtNo >= RT_COUNT)
        error = E_INVALID_PARAMETER;
    else
    {
        wW(card, LDB_RT_ADDRESS_REG, rtNo);
        card->ldbMrtRt = rtNo;
        error = SBIIB_Command(card, START_CARD);
    }

return (error);
}
