/****************************************************************************

   Copyright (c) 2000, Western Avionics LTD.             All Rights Reserved.

*****************************************************************************

   File          : sb_libry.c

   Author        : S.J.Buckley

   Date          : 11-01-2000

   Function      : SBIIB general purpose routines for all modes

   Update        : dd-mm-yy
   By            : S.J.Buckley
   Reason        :

****************************************************************************/
#include "sbiib.h"

/****************************************************************************

 Function  : SBIIB_RdStatus

 Purpose   : This function reads the SBIIB card status register

 Input(s)  : card       - handle for the card
             status     - address for status storage

 Output(s) : error      - Function status

****************************************************************************/
Error  SBIIB_RdStatus (CardHandle    *card,       /* Card handle           */
                       Uword         *status)     /* Card status           */
/***************************************************************************/
{
Error  error;
Ulong  tVal;

    error = E_NO_ERROR;

    /*
     If valid pointers, get the base register ID data into the structure.
    */
    if(card == NO_POINTER || status == NO_POINTER)
        error = E_INVALID_POINTER;
    else
        *status = rW(card, STATUS_REG);

return (error);
}

/****************************************************************************

 Function  : SBIIB_GetCardID

 Purpose   : This function gets the card serial number, checksum and rev.

 Input(s)  : card       - handle for the card
             info       - CardID info structure

 Output(s) : error      - Function status

****************************************************************************/
Error  SBIIB_GetCardID(CardHandle    *card,       /* Card handle           */
                       CardID        *info)       /* CardID structure      */
/***************************************************************************/
{
Error  error;
Ulong  tVal;
Sbyte  rev[SBIIB_DRVS_VERSION_SIZE] = SBIIB_DRVS_VERSION_INFO;

    error = E_NO_ERROR;

    /*
     If valid pointers, get the base register ID data into the structure.
    */
    if(card == NO_POINTER || info == NO_POINTER)
        error = E_INVALID_POINTER;
    else
    {
        tVal  = (Ulong)rW(card, SERIAL_NO_HI_REG);
        tVal  = (tVal << 16) & 0xFFFF0000;
        tVal |= (Ulong)rW(card, SERIAL_NO_LO_REG) & 0xFFFF;
        info->serialNo  = tVal;
        info->checksumH = rW(card, FIRMWARE_CHECKSUM_HI);
        info->checksumL = rW(card, FIRMWARE_CHECKSUM_LO);
        info->fwRev     = rW(card, FIRMWARE_REV_LEVEL);
        for(tVal=0; tVal != SBIIB_DRVS_VERSION_SIZE; tVal++)
            info->drvsRev[tVal] = rev[tVal]; 
    }

return (error);
}

/****************************************************************************

 Function  : SBIIB_SetErr

 Purpose   : This function converts the TXerrs into the error word and
             word number and saves it on the card.

 Input(s)  : card       - handle for the card
             offset     - Error address in card
             info       - Error structure

 Output(s) : error      - Function status

****************************************************************************/
Error  SBIIB_SetErr(CardHandle       *card,       /* Card handle           */
                    Ulong            offset,      /* Error address in card */
                    TXerrs           *info)       /* Error structure       */
/***************************************************************************/
{
Error  error;
Uword  eInfo, errWd, wordNo;

    error = E_NO_ERROR;

    if(card == NO_POINTER)
        error = E_INVALID_POINTER;
    else
    {
        /*
         If no error structure, return TX_NO_ERROR code.
        */
        if(info == NO_POINTER)
        {
            wW(card, offset, TX_NO_ERROR);
            wW(card, offset+2, 0);
        }

        /*
         Copy over the wordNo value and set the correct error word.
        */
        else
        {
            wordNo = info->wordNo  & (Uword)0x03FF;
            eInfo  = info->errInfo & (Uword)0x003F;

            switch (info->errType)
            {
                case TX_NO_ERROR:
                    errWd = info->errType;
                    wW(card, offset, errWd);
                    wW(card, offset+2, wordNo);
                    break;

                case TX_PARITY_ERROR:
                case TX_SYNC_ERROR:
                case TX_MANCHESTER_ERROR:
                case TX_BIT_COUNT_ERROR:
                    errWd = info->errType | eInfo;
                    wW(card, offset, errWd);
                    wW(card, offset+2, wordNo);
                    break;

                default:
                    error = E_INVALID_PARAMETER;
                    break;
            }
        }
    }

return (error);
}

/****************************************************************************

 Function  : SBIIB_GetErr

 Purpose   : This function fills the TXerrs with the error info on the card.

 Input(s)  : card       - handle for the card
             offset     - Error address in card
             info       - Error structure

 Output(s) : error      - Function status

****************************************************************************/
Error  SBIIB_GetErr(CardHandle       *card,       /* Card handle           */
                    Ulong            offset,      /* Error address in card */
                    TXerrs           *info)       /* Error structure       */
/***************************************************************************/
{
Error  error;
Uword  errWd;

    error = E_NO_ERROR;

    if(card == NO_POINTER)
        error = E_INVALID_POINTER;
    else
    {
        /*
         If error structure, get the errors.
        */
        if(info != NO_POINTER)
        {
            errWd = rW(card, offset);
            info->errType = errWd  & TX_ERROR_MASK;
            info->errInfo = errWd  & (Uword)0x003F;
            info->wordNo  = rW(card, offset+2);
        }
    }

return (error);
}

/****************************************************************************

 Function  : SBIIB_RdTtag

 Purpose   : This function reads in the Time tag words from the card and
             then formats the values in days, hours etc.

 Input(s)  : card       - handle for the card
             offset     - TTag address in card
             time       - TimeTag structure

 Output(s) : error      - Function status

****************************************************************************/
Error  SBIIB_RdTtag(CardHandle       *card,       /* Card handle           */
                    Ulong            offset,      /* TTag card address     */
                    TimeTag          *time)       /* TimeTag structure     */
/***************************************************************************/
{
Error  error;
Uword  hgh, mid, low, tck;

    if(card == NO_POINTER || time == NO_POINTER)
        error = E_INVALID_POINTER;
    else
    {
        hgh   = rW(card, offset);
        mid   = rW(card, offset+2);
        low   = rW(card, offset+4);
        tck   = rW(card, offset+6);
        error = SBIIB_FormatTtag(hgh, mid, low, tck, time);
    }

return (error);
}

/****************************************************************************

 Function  : SBIIB_FormatTtag

 Purpose   : Formats the timetag values and saves them in TimeTag structure.

 Input(s)  : hgh        - TTag HGH word
             mid        - TTag MID word
             low        - TTag LOW word
             tck        - TTag TCK word
             time       - TimeTag structure

 Output(s) : error      - Function status

****************************************************************************/
Error  SBIIB_FormatTtag(Uword      hgh,           /* HGH word              */
                        Uword      mid,           /* MID word              */
                        Uword      low,           /* LOW word              */
                        Uword      tck,           /* TCK word              */
                        TimeTag    *time)         /* TimeTag structure     */
/***************************************************************************/
{
Error error;

    error = E_NO_ERROR;

    if(time == NO_POINTER)
        error = E_INVALID_POINTER;
    else
    {
        time->ttagH = hgh;
        time->ttagM = mid;
        time->ttagL = low;
        time->ttagT = tck;

        time->lock    = (Uword)((hgh >> 15) & 1);
        time->days    = (Uword)((((hgh >> 12) & 3) * 100) + ((hgh >> 5) & 0x7F));
        time->hours   = (Uword)(hgh & 0x1F);
        time->minutes = (Uword)((mid >> 6) & 0x3F);
        time->seconds = (Uword)(mid & 0x3F);
        time->msecs   = (Uword)(low & 0x3FF);
        time->halfuS  = (Uword)(tck & 1);
        time->micros  = (Uword)((tck >> 1) & 0x3FF);
    }

return (error);
}

/****************************************************************************

 Function  : SBIIB_Command

 Purpose   : This function executes a cammand on the SBIIB card.

 Input(s)  : card       - handle for the card
             command    - Command to execute

 Output(s) : error      - Function status

****************************************************************************/
Error  SBIIB_Command(CardHandle *card,           /* Card handle            */
                     Uword      command)         /* Command to execute     */
/***************************************************************************/
{
Error  error;

    error = E_NO_ERROR;

    if(card == NO_POINTER)
        error = E_INVALID_POINTER;

    /*
     Test if command register is clear
    */
    if(error == E_NO_ERROR)
    {
        if(rW(card, COMMAND_REG) != 0)
            error = E_CANNOT_EXECUTE_COMMAND;
    }

    /*
     Set command and assert request
    */
    if(error == E_NO_ERROR)
    {
        wW(card, COMMAND_REG, command);
        wW(card, CONTROL_REG, EXECUTE_COMMAND);
    }

return (error);
}

/****************************************************************************

 Function  : SBIIB_LoadClock

 Purpose   : Loads the clock with the value in the ClockInit structure.

 Input(s)  : card       - handle for the card
             info       - ClockInit structure

 Output(s) : error      - Function status

****************************************************************************/
Error  SBIIB_LoadClock(CardHandle    *card,       /* Card handle           */
                       ClockInit     *info)       /* ClockInit structure   */
/***************************************************************************/
{
Error error;
Uword valH, valL, dcs, dys, hrs, mns, scs, tmp;

    error = E_NO_ERROR;

    if(card == NO_POINTER || info == NO_POINTER)
        error = E_INVALID_POINTER;
    else
    {
        /*
         If free running, check that parameters are correct. If true, convert
         the  parameters into the  2 word  format required by the  card for a
         clock load.
        */
        if(info->mode == CLOCK_FREE_RUNNING)
        {
            if(info->leapYear > 3 || info->days > 366 ||
               info->hours > 23   || info->minutes > 59 || info->seconds > 59)
                error = E_INVALID_PARAMETER;
            else
            {
                dcs   = (Uword)(info->days/100);
                dys   = (Uword)(info->days % 100);
                tmp   = (Uword)(dys/10);
                dys   = (Uword)(dys%10);
                dys  |= (Uword)(tmp << 4);
                tmp   = (Uword)(info->hours/10);
                hrs   = (Uword)(info->hours % 10);
                hrs  |= (Uword)(tmp << 4);
                tmp   = (Uword)(info->minutes/10);
                mns   = (Uword)(info->minutes % 10);
                mns  |= (Uword)(tmp << 4);
                tmp   = (Uword)(info->seconds/10);
                scs   = (Uword)(info->seconds % 10);
                scs  |= (Uword)(tmp << 4);

                valH  = (Uword)(info->leapYear << 14);
                valH |= (Uword)(dcs << 12);
                valH |= (Uword)(dys << 5);
                valH |= (Uword)(hrs >> 2);
                valL  = (Uword)(hrs << 10);
                valL |= (Uword)(mns << 7);
                valL |= (Uword)scs;
            }
        }

        /*
         If set for demodulating IRIG-B, check leap year value is good. If true,
         set the leap year bits and set the 2 words for demodulation mode.
        */
        else
        if(info->mode == CLOCK_DEMODULATE_IRIG)
        {
            if(info->leapYear > 3)
                error = E_INVALID_PARAMETER;
            else
            {
                valH = (Uword)((info->leapYear << 14) | 0x3FFF);
                valL = 0xFFFF;
            }
        }
        else
            error = E_INVALID_PARAMETER;

        /*
         If no error, set the values in the base register and assert command.
        */
        if(error == E_NO_ERROR)
        {
            wW(card, LOAD_CLK_HI_REG, valH);
            wW(card, LOAD_CLK_LO_REG, valL);
            error = SBIIB_Command(card, LOAD_CLOCK);
        }
    }

return (error);
}

/****************************************************************************

 Function  : SBIIB_RdBinQ

 Purpose   : This function reads an entry of the BIN Queue.

 Input(s)  : card    - handle for the card
             qData   - LDB_BIN_Qdata structure for storing result

 Output(s) : error   - Function status

****************************************************************************/
Error  SBIIB_RdBinQ(CardHandle       *card,       /* Card handle           */
                    LDB_BIN_Qdata    *qData)      /* Bin value storage     */
/***************************************************************************/
{
Error  error;
Ulong  aVal, tVal, offset, cVal;

    error  = E_NO_ERROR;
    offset = qData->offset * BIN_Q_ELEMENT_SZ;

    /*
     Test if parameters values are good.
    */
    if(card == NO_POINTER || qData == NO_POINTER)
        error = E_INVALID_POINTER;
    else
    if(card->mode != LDB_BCT_MODE && card->mode != LDB_MRT_MODE)
        error = E_INVALID_MODE;
    else
    if(offset > (BIN_Q_SZ - BIN_Q_ELEMENT_SZ))
        error = E_INVALID_PARAMETER;

    /*
     If clear the queue, reset offset and clear all queue data.
    */
    else
    if(qData->action == CLEAR_Q)
    {
        qData->offset = 0;

        for(aVal=BIN_Q, tVal=0; tVal != BIN_Q_SZ; tVal+=2)
            wW(card, aVal+tVal, 0);
        qData->result = NO_Q_ELEMENT;
    }

    /*
     Test for  element in queue.  If true, clear it, read in the time tag and
     read  in the  data  HI and LO values.  Finally increment  the  Q pointer
     accounting for wrap-round.
    */
    else
    if(qData->action == READ_Q || qData->action == READ_Q_AND_CLEAR)
    {
        aVal = BIN_Q + offset;

        if(rW(card, aVal+14) == 0)
            qData->result = NO_Q_ELEMENT;
        else
        {
            qData->result = GOT_Q_ELEMENT;
            if(qData->action == READ_Q_AND_CLEAR)
                wW(card, aVal+14, 0);
            cVal  = (Ulong)rW(card, aVal+8);
            cVal  = (cVal << 16) & 0xFFFF0000;
            cVal |= (Ulong)rW(card, aVal+10) & 0xFFFF;
            qData->data    = cVal;
            qData->altData = rW(card, aVal+12);
            error = SBIIB_RdTtag(card, aVal, &qData->time);
            offset += BIN_Q_ELEMENT_SZ;
            offset &= (BIN_Q_SZ - BIN_Q_ELEMENT_SZ);
            qData->offset = offset/BIN_Q_ELEMENT_SZ;
        }
    }
    else
        error = E_INVALID_PARAMETER;

return (error);
}
