/****************************************************************************

   Copyright (c) 2000, Western Avionics LTD.             All Rights Reserved.

*****************************************************************************

   File          : sb_setup.c

   Author        : S.J.Buckley

   Date          : 11-01-2000

   Function      : SBIIB card setup functions.

   Update        : dd-mm-yy
   By            : S.J.Buckley
   Reason        :

****************************************************************************/
#include "sbiib.h"

/****************************************************************************

 Function  : SBIIB_Setup

 Purpose   : This function initialises the SBIIB card base registers etc.

 Input(s)  : card       - handle for the card
             mode       - Card mode
             info       - SetupInfo strucure information for base registers

 Output(s) : error      - Function status

****************************************************************************/
Error  SBIIB_Setup(CardHandle  *card,             /* Card handle           */
                   Uword       mode,              /* Card mode             */
                   SetupInfo   *info)             /* Setup info            */
/***************************************************************************/
{
Error  error;
Uword  tVal, i;

    error = E_NO_ERROR;

    if(card == NO_POINTER)
        error = E_INVALID_POINTER;

    /********************************************************************
    *                  Initialise the card base registers               *
    ********************************************************************/
    if(error == E_NO_ERROR)
    {
        /*
         Initialise the base registers with defaults to begin
        */
        wW(card, COMMAND_REG, 0);
        wW(card, STATUS_REG, 0);
        wW(card, BACKGROUND_REG, (Uword)LDB_PROGRAM);
        wW(card, RT_SIM_PTR, (Uword)RT_SIM_TABLE);
        wW(card, HI_PRI_Q_PTR, (Uword)HI_PRI_Q);
        wW(card, LO_PRI_Q_PTR, (Uword)LO_PRI_Q);
        wW(card, BIN_Q_PTR, (Uword)BIN_Q);
        wW(card, BIN_Q_EVENT, BIN_Q_EVENT_DEF);
        wW(card, VME_IRQ_1, VME_IRQ_1_DEF);
        wW(card, VME_IRQ_2, VME_IRQ_2_DEF);
        wW(card, IRQ_SELECTION, IRQ_SELECTION_DEF);
        wW(card, GAP_TIME_REG, GAP_TIME_REG_DEF);
        wW(card, RT_RESPONSE_TIME, RT_RESPONSE_TIME_DEF);
        wW(card, RT_RESPONSE_TIMEOUT, RT_RESPONSE_TIMEOUT_DEF);
        wW(card, BUS_SELECT_REG, BUS_SELECT_REG_DEF);
        wW(card, BUS_MON_SETUP_PTR, (Uword)MON_SETUP);
        wW(card, LOAD_CLK_HI_REG, 0);
        wW(card, LOAD_CLK_LO_REG, 0);
        wW(card, INTER_WORD_TIMEOUT, INTER_WORD_TIMEOUT_DEF);
        wW(card, WATCHDOG_TIMEOUT, WATCHDOG_TIMEOUT_DEF);
        wW(card, WATCHDOG_CHANGE, 0);
        wW(card, POLLING_TIMEOUT, POLLING_TIMEOUT_DEF);
        wW(card, TX_AMPLITUDE_REG, TX_AMPLITUDE_DEF);
        wW(card, TX_PRI_REG, TX_PRI_REG_DEF);
        wW(card, TX_SEC_REG, TX_SEC_REG_DEF);
        wW(card, RX_PRI_REG, RX_PRI_REG_DEF);
        wW(card, RX_SEC_REG, RX_SEC_REG_DEF);
        wW(card, LDB_RT_ADDRESS_REG, LDB_MRT_RT_DEF);
        wW(card, WRAP_ROUND_HI_REG, WRAP_ROUND_HI_REG_DEF);
        wW(card, WRAP_ROUND_LO_REG, WRAP_ROUND_LO_REG_DEF);

        /*
         If setup data, store defined data in base registers.
        */
        if(info != NO_POINTER)
        {
            /*
             Test for BITS PER WORD and if true and valid,
             set the value.
            */
            if(info->whatToSet & F_SET_BITS_PER_WORD)
            {
                tVal = info->settings[SET_BITS_PER_WORD];
                if(tVal != DATA_BITS_24 && tVal != DATA_BITS_40)
                    error = E_INVALID_PARAMETER;
            }
            else
                tVal = 0;
    
            /*
             Test for GAP TIME and if true and valid, set the value.
            */
            if(info->whatToSet & F_SET_GAP_TIME)
            {
                if(info->settings[SET_GAP_TIME] > GAP_TIME_MAX)
                    error = E_INVALID_PARAMETER;
                tVal |= info->settings[SET_GAP_TIME];
            }
            else
                tVal |= GAP_TIME_REG_DEF;
    
            /*
             Output the GAP TIME word.
            */
            wW(card, GAP_TIME_REG, tVal);
    
            /*
             Test for CHECKSUM on-off and if true and valid,
             set the value.
            */
            if(info->whatToSet & F_SET_CHECKSUM_ON_OFF)
            {
                if(info->settings[SET_CHECKSUM_ON_OFF] != CHECKSUM_ON &&
                   info->settings[SET_CHECKSUM_ON_OFF] != CHECKSUM_OFF)
                    error = E_INVALID_PARAMETER;
                tVal = info->settings[SET_CHECKSUM_ON_OFF];
            }
            else
                tVal = 0;
    
            /*
             Test for GLOBAL errors on-off and if true and valid,
             set the value.
            */
            if(info->whatToSet & F_SET_GLOBAL_ERRS_ON_OFF)
            {
                if(info->settings[SET_GLOBAL_ERRS_ON_OFF] != GLOBAL_ERRS_ON &&
                   info->settings[SET_GLOBAL_ERRS_ON_OFF] != GLOBAL_ERRS_OFF)
                    error = E_INVALID_PARAMETER;
                tVal |= info->settings[SET_GLOBAL_ERRS_ON_OFF];
            }

            /*
             Test for BUS SELECT and if true set the value.
            */
            if(info->whatToSet & F_SET_BUS_SELECT)
            {
                tVal |= info->settings[SET_BUS_SELECT] & (Uword)0x000F;
            }
            else
                tVal |= BUS_SELECT_REG_DEF;
    
            /*
             Output the BUS SELECT word.
            */
            wW(card, BUS_SELECT_REG, tVal);

            /*
             Test for TX AMPLITUDE and if true set the value.
            */
            if(info->whatToSet & F_SET_TX_AMPLITUDE)
                tVal = info->settings[SET_TX_AMPLITUDE] & (Uword)0x00FF;
            else
                tVal = TX_AMPLITUDE_DEF;
    
            /*
             Test for SET CLOCK MODE and if true and valid,
             set the value.
            */
            if(info->whatToSet & F_SET_CLOCK_IRIG_FREE)
            {
                i = info->settings[SET_CLOCK_IRIG_FREE];
                if(i != CLOCK_FREE_RUNNING && i != CLOCK_DEMODULATE_IRIG)
                    error = E_INVALID_PARAMETER;
                tVal |= i;
            }
    
            /*
             Output the TX AMPLITUDE value.
            */
            wW(card, TX_AMPLITUDE_REG, tVal);
    
            /*
             Test for SET CONNECTORS and if true and valid,
             set the values.
            */
            if(info->whatToSet & F_SET_CONNECTORS)
            {
                tVal = info->settings[SET_TX_PRI_CONNECTOR];
                if(tVal != J1_CONNECTOR && tVal != J2_CONNECTOR &&
                   tVal != J3_CONNECTOR && tVal != J4_CONNECTOR)
                    error = E_INVALID_PARAMETER;
                tVal = info->settings[SET_TX_SEC_CONNECTOR];
                if(tVal != J1_CONNECTOR && tVal != J2_CONNECTOR &&
                   tVal != J3_CONNECTOR && tVal != J4_CONNECTOR)
                    error = E_INVALID_PARAMETER;
                tVal = info->settings[SET_RX_PRI_CONNECTOR];
                if(tVal != J1_CONNECTOR && tVal != J2_CONNECTOR &&
                   tVal != J3_CONNECTOR && tVal != J4_CONNECTOR)
                    error = E_INVALID_PARAMETER;
                tVal = info->settings[SET_RX_SEC_CONNECTOR];
                if(tVal != J1_CONNECTOR && tVal != J2_CONNECTOR &&
                   tVal != J3_CONNECTOR && tVal != J4_CONNECTOR)
                    error = E_INVALID_PARAMETER;
    
                wW(card, TX_PRI_REG, info->settings[SET_TX_PRI_CONNECTOR]);
                wW(card, TX_SEC_REG, info->settings[SET_TX_SEC_CONNECTOR]);
                wW(card, RX_PRI_REG, info->settings[SET_RX_PRI_CONNECTOR]);
                wW(card, RX_SEC_REG, info->settings[SET_RX_SEC_CONNECTOR]);
            }

            /*
             Now test and set the remaining parameters.
            */
            if(info->whatToSet & F_SET_RESPONSE_TIME)
                wW(card, RT_RESPONSE_TIME, info->settings[SET_RESPONSE_TIME]);
            if(info->whatToSet & F_SET_RESPONSE_TIMEOUT)
                wW(card, RT_RESPONSE_TIMEOUT, info->settings[SET_RESPONSE_TIMEOUT]);
            if(info->whatToSet & F_SET_INTER_WORD_TIMEOUT)
                wW(card, INTER_WORD_TIMEOUT, info->settings[SET_INTER_WORD_TIMEOUT]);
            if(info->whatToSet & F_SET_WATCHDOG_TIMEOUT)
                wW(card, WATCHDOG_TIMEOUT, info->settings[SET_WATCHDOG_TIMEOUT]);
            if(info->whatToSet & F_SET_POLLING_TIMEOUT)
                wW(card, POLLING_TIMEOUT, info->settings[SET_POLLING_TIMEOUT]);
            if(info->whatToSet & F_SET_VME_IRQ_1)
                wW(card, VME_IRQ_1, info->settings[SET_VME_IRQ_1]);
            if(info->whatToSet & F_SET_VME_IRQ_2)
                wW(card, VME_IRQ_2, info->settings[SET_VME_IRQ_2]);
            if(info->whatToSet & F_SET_IRQ_SELECTION)
                wW(card, IRQ_SELECTION, info->settings[SET_IRQ_SELECTION]);
        }
    }


    /*
     Now initialise the card for the particular mode.
    */
    if(error == E_NO_ERROR)
    {
        switch(mode)
        {
            case LDB_BCT_MODE:
                error = SBIIB_SetupLdbBc(card);
                break;

            case LDB_MRT_MODE:
                error = SBIIB_SetupLdbRt(card);
                break;

            case BUS_MON_MODE:
                error = SBIIB_SetupMon(card);
                break;

            case MDM_MRT_MODE:
                error = SBIIB_SetupMdmRt(card);
                break;

            default:
                error = E_INVALID_PARAMETER;
                break;
        }
    }

    /*
     Now put the card into the desired mode
    */
    if(error == E_NO_ERROR)
        error = SBIIB_Command(card, mode);
    
return (error);
}

/****************************************************************************

 Function  : SBIIB_RdSetup

 Purpose   : This function reads the SBIIB base register information.

 Input(s)  : card       - handle for the card
             info       - Array of setup information for base registers

 Output(s) : error      - Function status

****************************************************************************/
Error  SBIIB_RdSetup(CardHandle  *card,           /* Card handle           */
                     SetupInfo   *info)           /* Setup info            */
/***************************************************************************/
{
Error error;
Uword tVal;

    error = E_NO_ERROR;

    /*
     If good handle and SetupInfo, read in the values into the array.
    */
    if(card == NO_POINTER || info == NO_POINTER)
        error = E_INVALID_POINTER;
    else
    {
        tVal = rW(card, GAP_TIME_REG);
        info->settings[SET_BITS_PER_WORD]      = (Uword)(tVal & DATA_BITS_40);
        info->settings[SET_GAP_TIME]           = (Uword)(tVal & GAP_TIME_MAX);

        tVal = rW(card, BUS_SELECT_REG);
        info->settings[SET_CHECKSUM_ON_OFF]    = (Uword)(tVal & CHECKSUM_OFF);
        info->settings[SET_GLOBAL_ERRS_ON_OFF] = (Uword)(tVal & GLOBAL_ERRS_OFF);
        info->settings[SET_BUS_SELECT]         = (Uword)(tVal & 0x000F);

        tVal = rW(card, TX_AMPLITUDE_REG);
        info->settings[SET_TX_AMPLITUDE]       = (Uword)(tVal & 0x00FF);
        info->settings[SET_CLOCK_IRIG_FREE]    = (Uword)(tVal & CLOCK_DEMODULATE_IRIG);

        info->settings[SET_TX_PRI_CONNECTOR]   = rW(card, TX_PRI_REG);
        info->settings[SET_TX_SEC_CONNECTOR]   = rW(card, TX_SEC_REG);
        info->settings[SET_RX_PRI_CONNECTOR]   = rW(card, RX_PRI_REG);
        info->settings[SET_RX_SEC_CONNECTOR]   = rW(card, RX_SEC_REG);
        info->settings[SET_RESPONSE_TIME]      = rW(card, RT_RESPONSE_TIME);
        info->settings[SET_RESPONSE_TIMEOUT]   = rW(card, RT_RESPONSE_TIMEOUT);
        info->settings[SET_INTER_WORD_TIMEOUT] = rW(card, INTER_WORD_TIMEOUT);
        info->settings[SET_WATCHDOG_TIMEOUT]   = rW(card, WATCHDOG_TIMEOUT);
        info->settings[SET_POLLING_TIMEOUT]    = rW(card, POLLING_TIMEOUT);
        info->settings[SET_TX_AMPLITUDE]       = rW(card, TX_AMPLITUDE_REG);
        info->settings[SET_VME_IRQ_1]          = rW(card, VME_IRQ_1);
        info->settings[SET_VME_IRQ_2]          = rW(card, VME_IRQ_2);
        info->settings[SET_IRQ_SELECTION]      = rW(card, IRQ_SELECTION);
    }

return (error);
}

/****************************************************************************

 Function  : SBIIB_SetupLdbRt

 Purpose   : This function initialises the SBIIB card for LDB_MRT mode.

 Input(s)  : card       - handle for the card

 Output(s) : error      - Function status

****************************************************************************/
Error  SBIIB_SetupLdbRt(CardHandle  *card)        /* Card handle           */
/***************************************************************************/
{
Error  error;
Ulong  bVal, sVal, addr;

    error = E_NO_ERROR;

    /*
     Initialise the RT sim table
    */
    addr = RT_SIM_TABLE;
    bVal = DATA_BLOCKS;
    sVal = DATA_BLOCKS + DATA_BLOCK_SZ;
    
    wW(card, addr+LDB_SIM_MODEWD, LDB_MRT_PASSIVE);
    wW(card, addr+LDB_SIM_STATEM, WFI);
    wW(card, addr+LDB_SIM_IRQWDA, 0x6DB6);
    wW(card, addr+LDB_SIM_IRQWDB, 0x01F6);
    wW(card, addr+LDB_SIM_TXBFHI, (Uword)((bVal >> 16) & 0x000F));
    wW(card, addr+LDB_SIM_TXBFLO, (Uword)(bVal & 0xFFFF));
    wW(card, addr+LDB_SIM_RXBFHI, (Uword)((sVal >> 16) & 0x000F));
    wW(card, addr+LDB_SIM_RXBFLO, (Uword)(sVal & 0xFFFF));
    wW(card, addr+LDB_SIM_TTAGHI, 0);
    wW(card, addr+LDB_SIM_TTAGMD, 0);
    wW(card, addr+LDB_SIM_TTAGLO, 0);
    wW(card, addr+LDB_SIM_TTAGTK, 0);
    wW(card, addr+LDB_SIM_LCMDHI, 0);
    wW(card, addr+LDB_SIM_LCMDLO, 0);
    wW(card, addr+LDB_SIM_LRESHI, 0);
    wW(card, addr+LDB_SIM_LRESLO, 0);
    wW(card, addr+LDB_SIM_GAHERR, TX_NO_ERROR);
    wW(card, addr+LDB_SIM_GAHWRD, 0);
    wW(card, addr+LDB_SIM_INDERR, TX_NO_ERROR);
    wW(card, addr+LDB_SIM_INDWRD, 0);
    wW(card, addr+LDB_SIM_IWDERR, TX_NO_ERROR);
    wW(card, addr+LDB_SIM_IWDWRD, 0);
    wW(card, addr+LDB_SIM_SRQERR, TX_NO_ERROR);
    wW(card, addr+LDB_SIM_SRQWRD, 0);
    wW(card, addr+LDB_SIM_FRCERR, TX_NO_ERROR);
    wW(card, addr+LDB_SIM_FRCWRD, 0);
    wW(card, addr+LDB_SIM_RESVD1, 0);
    wW(card, addr+LDB_SIM_RESVD2, 0);
    wW(card, addr+LDB_SIM_RESVD3, 0);
    wW(card, addr+LDB_SIM_RESVD4, 0);
    wW(card, addr+LDB_SIM_RESVD5, 0);
    wW(card, addr+LDB_SIM_RESVD6, 0);
   
    /*
     Initialise all the data buffer
    */
    addr = DATA_BLOCKS;
    wW(card, addr+LDB_DATA_TTAGHI, 0);
    wW(card, addr+LDB_DATA_TTAGMD, 0);
    wW(card, addr+LDB_DATA_TTAGLO, 0);
    wW(card, addr+LDB_DATA_TTAGTK, 0);
    wW(card, addr+LDB_DATA_BUFFSZ, 1000);
    wW(card, addr+LDB_DATA_STATUS, 0);
    wW(card, addr+LDB_DATA_WORDCT, 0);
    wW(card, addr+LDB_DATA_CHKSUM, 0);
    wW(card, addr+LDB_DATA_NEXTHI, (Uword)((addr >> 16) & 0x000F));
    wW(card, addr+LDB_DATA_NEXTLO, (Uword)(addr & 0xFFFF));

    /*
     Clear the queues
    */
    for(addr=BIN_Q, bVal=0; bVal!=BIN_Q_SZ; bVal+=2)
        wW(card, addr+bVal, 0);
    for(addr=HI_PRI_Q, bVal=0; bVal!=HI_PRI_Q_SZ; bVal+=2)
        wW(card, addr+bVal, 0);
    for(addr=LO_PRI_Q, bVal=0; bVal!=LO_PRI_Q_SZ; bVal+=2)
        wW(card, addr+bVal, 0);
 
return (error);
}

/****************************************************************************

 Function  : SBIIB_SetupLdbBc

 Purpose   : This function initialises the SBIIB card for LDB_BCT mode.

 Input(s)  : card       - handle for the card

 Output(s) : error      - Function status

****************************************************************************/
Error  SBIIB_SetupLdbBc(CardHandle  *card)        /* Card handle           */
/***************************************************************************/
{
Error  error;
Ulong  sVal, bVal, addr;

    error = E_NO_ERROR;

    /*
     Fill program area with HALT and initialise with simple frame
    */
    wW(card, BACKGROUND_REG, LDB_PROGRAM);
    for(addr=LDB_PROGRAM, sVal=0; sVal!=LDB_PROGRAM_SZ; sVal+=2)
        wW(card, addr+sVal, LDB_HALT);
    
    wW(card, addr,    LDB_LOOP);
    wW(card, addr+2,  0);
    wW(card, addr+4,  LDB_INITF);
    wW(card, addr+6,  0);
    wW(card, addr+8,  40000);
    wW(card, addr+10, LDB_SWPSE);
    wW(card, addr+12, LDB_NOP3);
    wW(card, addr+14, 0xAAAA);
    wW(card, addr+16, 0xBBBB);
    wW(card, addr+18, LDB_TRIGOUT);
    wW(card, addr+20, 0x8001);
    wW(card, addr+22, LDB_SMB);
    wW(card, addr+24, (Uword)MDB_START);
    wW(card, addr+26, (Uword)(addr+24));
    wW(card, addr+28, LDB_BRA);
    wW(card, addr+30, 0xFFEE);
    wW(card, addr+32, LDB_HALT);

    /*
     Initialise all the MDBs
    */
    for(sVal=0; sVal!=MDB_COUNT; sVal++)
    {
        addr = MDB_START   + (sVal * MDB_SZ);
        bVal = DATA_BLOCKS + (sVal * DATA_BLOCK_SZ);
    
        wW(card, addr+LDB_MDB_NUMBER, (Uword)(sVal+1));
        wW(card, addr+LDB_MDB_IRQCON, 6);
        wW(card, addr+LDB_MDB_CMNDHI, 0x8009);
        wW(card, addr+LDB_MDB_CMNDLO, 0x8000);
        wW(card, addr+LDB_MDB_TXERRS, TX_NO_ERROR);
        wW(card, addr+LDB_MDB_TXERWD, 0);
        wW(card, addr+LDB_MDB_DATAHI, (Uword)((bVal >> 16) & 0x000F));
        wW(card, addr+LDB_MDB_DATALO, (Uword)(bVal & 0xFFFF));
        wW(card, addr+LDB_MDB_DATAWC, 0x4001);
        wW(card, addr+LDB_MDB_MEVENT, 0);
        wW(card, addr+LDB_MDB_PATTHI, 0x0000);
        wW(card, addr+LDB_MDB_PATTLO, 0x0000);
        wW(card, addr+LDB_MDB_MASKHI, 0x00FF);
        wW(card, addr+LDB_MDB_MASKLO, 0xFFFF);
        wW(card, addr+LDB_MDB_GAPTHI, 0);
        wW(card, addr+LDB_MDB_GAPTLO, 1000);
    }
    
    /*
     Initialise all the data buffers
    */
    for(sVal=0; sVal!=DATA_BLOCK_COUNT; sVal++)
    {
        addr = DATA_BLOCKS + (sVal * DATA_BLOCK_SZ);
        wW(card, addr+LDB_DATA_TTAGHI, 0);
        wW(card, addr+LDB_DATA_TTAGMD, 0);
        wW(card, addr+LDB_DATA_TTAGLO, 0);
        wW(card, addr+LDB_DATA_TTAGTK, 0);
        wW(card, addr+LDB_DATA_BUFFSZ, 1000);
        wW(card, addr+LDB_DATA_STATUS, 0);
        wW(card, addr+LDB_DATA_WORDCT, 0);
        wW(card, addr+LDB_DATA_CHKSUM, 0);
        wW(card, addr+LDB_DATA_NEXTHI, (Uword)((addr >> 16) & 0x000F));
        wW(card, addr+LDB_DATA_NEXTLO, (Uword)(addr & 0xFFFF));
    }

    /*
     Clear the queues
    */
    for(addr=BIN_Q, sVal=0; sVal!=BIN_Q_SZ; sVal+=2)
        wW(card, addr+sVal, 0);
    for(addr=HI_PRI_Q, sVal=0; sVal!=HI_PRI_Q_SZ; sVal+=2)
        wW(card, addr+sVal, 0);
    for(addr=LO_PRI_Q, sVal=0; sVal!=LO_PRI_Q_SZ; sVal+=2)
        wW(card, addr+sVal, 0);

return (error);
}

/****************************************************************************

 Function  : SBIIB_SetupMon

 Purpose   : This function initialises the SBIIB card for BUS_MON mode.

 Input(s)  : card       - handle for the card

 Output(s) : error      - Function status

****************************************************************************/
Error  SBIIB_SetupMon(CardHandle  *card)          /* Card handle           */
/***************************************************************************/
{
Error  error;
Ulong  sVal;

    error = E_NO_ERROR;

    /*
     Initialise MON setup registers.
    */
    wW(card, MON_BIT_MODE, MON_BIT_MODE_DEF);
    wW(card, MON_BUS_DEFINITION, MON_BUS_DEFINITION_DEF);
    wW(card, MON_BUS_DEF_DISABLE, MON_BUS_DEF_DISABLE_DEF);
    wW(card, MON_PATTERN_HGH, MON_PATTERN_HGH_DEF);
    wW(card, MON_PATTERN_MID, MON_PATTERN_MID_DEF);
    wW(card, MON_PATTERN_LOW, MON_PATTERN_LOW_DEF);
    wW(card, MON_MASK_HGH, MON_MASK_HGH_DEF);
    wW(card, MON_MASK_MID, MON_MASK_MID_DEF);
    wW(card, MON_MASK_LOW, MON_MASK_LOW_DEF);
    wW(card, MON_ERROR_EVENT, MON_ERROR_EVENT_DEF);
    wW(card, MON_SELECT_LUPT_PTR, MON_SELECT_LUPT_PTR_DEF);
    wW(card, MON_PTC_HGH, MON_PTC_HGH_DEF);
    wW(card, MON_PTC_LOW, MON_PTC_LOW_DEF);

    sVal = MON_STACK_START_ADDR;

    wW(card, MON_STACK_START_HI, (Uword)((sVal >> 16) & 0xFFFF));
    wW(card, MON_STACK_START_LO, (Uword)(sVal & 0xFFFF));

    sVal = MON_STACK_STOP_ADDR;

    wW(card, MON_STACK_STOP_HI, (Uword)((sVal >> 16) & 0xFFFF));
    wW(card, MON_STACK_STOP_LO, (Uword)(sVal & 0xFFFF));

    /*
     Enable all RTs for capture.
    */
    for(sVal=0; sVal!= MON_RT_SELECT_TABLE_SZ; sVal+=2)
        wW(card, MON_RT_SELECT_TABLE+sVal, MON_RT_SELECTIVE_ON);
 
return (error);
}

/****************************************************************************

 Function  : SBIIB_SetupMdmRt

 Purpose   : This function initialises the SBIIB card for MDM_MRT mode.

 Input(s)  : card       - handle for the card

 Output(s) : error      - Function status

****************************************************************************/
Error  SBIIB_SetupMdmRt(CardHandle  *card)        /* Card handle           */
/***************************************************************************/
{
Error  error;

    error = E_NO_ERROR;
 
return (error);
}
